#region references

using System;
using System.Data;
using System.Collections;
using Method = System.Reflection.MethodBase;
using System.Drawing.Printing;
using System.Text;
using Microsoft.Reporting.WinForms;
using System.Reflection;
using System.IO;
using BROKEN = gov.va.med.vbecs.Common.VbecsBrokenRules.InappropriateTransfusionRequestReport;
using TABLE = gov.va.med.vbecs.Common.VbecsTables;
using ARTIFICIAL = gov.va.med.vbecs.Common.DatabaseConstants.ArtificialColumnNames;

#endregion

namespace gov.va.med.vbecs.BOL
{
	/// <summary>
	/// Summary description for ExceptionReport.
	/// </summary>
	public class ExceptionReport : BOL.VbecsReport
	{
		private Common.ExceptionType _exceptionType;
		private DataRow _exceptionData;
		private string _exceptionComment;
		private DateTime _exceptionDate;
		private string _exceptionUser;
        private Hashtable _exceptionReports = new Hashtable();

		#region constructors

		///<Developers>
		///	<Developer>Krzysztof Dobranowski</Developer>
		///</Developers>'gov.va.med.vbecs.GUI.Reports'
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/7/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4207"> 
		///		<ExpectedInput>Valid exception type, comment and user</ExpectedInput>
		///		<ExpectedOutput>Valid ExceptionReport object</ExpectedOutput>
		///	</Case>
		///<Case type="1" testid ="2254"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// ExceptionReport class
		/// </summary>
		public ExceptionReport(Common.ExceptionType exceptionType, string comment, string user)
		{
			this._exceptionType = exceptionType;
			this._exceptionComment = comment.Trim();
			this._exceptionDate = BOL.VBECSDateTime.GetDivisionCurrentDateTime();
			this._exceptionUser = GetUserNameFromID(user);
		}

		///<Developers>
		///	<Developer>Rob Heiberger</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/31/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6638"> 
		///		<ExpectedInput>Valid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="6639"> 
		///		<ExpectedInput>Invalid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Blank Constructor
		/// </summary>
		public ExceptionReport()
		{
			SetInitialBrokenRules( BROKEN.FormClassName, BROKEN.RuleSets.BaseRuleSet );

		}

		#endregion


		#region properties

		///<Developers>
		///	<Developer>Krzysztof Dobranowski</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/7/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4209"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>Common.ExceptionType</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4210"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Type of exception report
		/// </summary>
		public Common.ExceptionType ExceptionType
		{
			get
			{
				return this._exceptionType;
			}
			set
			{
				this._exceptionType = value;
			}
		}

		///<Developers>
		///	<Developer>Krzysztof Dobranowski</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/7/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4211"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>Valid DataRow</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4212"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Exception details
		/// </summary>
		public DataRow ExceptionData
		{
			get
			{
				return this._exceptionData;
			}
			set
			{
				this._exceptionData = value;
			}
		}

		///<Developers>
		///	<Developer>Krzysztof Dobranowski</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/7/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4213"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>string</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4214"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Exception comment
		/// </summary>
		public string ExceptionComment
		{
			get
			{
				return this._exceptionComment;
			}
			set
			{
				this._exceptionComment = value;
			}
		}

		///<Developers>
		///	<Developer>Krzysztof Dobranowski</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/7/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4215"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>Valid DateTime</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4216"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Exception date and time
		/// </summary>
		public DateTime ExceptionDate
		{
			get
			{
				return this._exceptionDate;
			}
			set
			{
				this._exceptionDate = value;
			}
		}

		///<Developers>
		///	<Developer>Krzysztof Dobranowski</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/7/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4217"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>Valid string</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4218"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Exception user
		/// </summary>
		public string ExceptionUser
		{
			get
			{
				return this._exceptionUser;
			}
			set
			{
				this._exceptionUser = value;
			}
		}

		#endregion


		#region methods
        /// <summary>
        /// Get stored procedure name for given exception type
        /// </summary>
        /// <param name="exceptionType"></param>
        /// <returns></returns>
        public static string GetStoredProcName(Common.ExceptionType exceptionType)
        {
            string storedProcName = string.Empty;

            switch(exceptionType)
            {
                case Common.ExceptionType.ABORh_confirmation_inconclusive_unit_quarantined:
                case Common.ExceptionType.ABORh_confirmation_does_not_match_login_ABORh:
                    storedProcName = Common.VbecsStoredProcs.InsertExceptionUnitAboRH.StoredProcName;
                    break;
                case Common.ExceptionType.Unit_ER_issued_testing_problem:
                    storedProcName = Common.VbecsStoredProcs.InsertExceptionEmergencyIssueUnitTest.StoredProcName;
                    break;
                case Common.ExceptionType.Expired_antisera_used:
                    storedProcName = Common.VbecsStoredProcs.InsertExceptionQC.StoredProcName;
                    break;
                case Common.ExceptionType.ABORh_discrepancy:
                    storedProcName = Common.VbecsStoredProcs.InsertExceptionPatientAboRHDiscrepancy.StoredProcName;
                    break;
                case Common.ExceptionType.Expired_task_processed:
                    storedProcName = Common.VbecsStoredProcs.InsertExceptionExpiredTask.StoredProcName;
                    break;  
                case Common.ExceptionType.Discrepant_patient_antigen_typing:
                    storedProcName = Common.VbecsStoredProcs.InsertExceptionPatientAntigenDiscrepancy.StoredProcName;
                    break;
                case Common.ExceptionType.Crossmatch_incompatible_Give_only_with_MD_approval:
                    storedProcName = Common.VbecsStoredProcs.InsertExceptionUnitSelection.StoredProcName;
                    break;          
            }

            if (string.IsNullOrWhiteSpace(storedProcName))
            {
                throw new Exception("Stored procedure not found for Exception Type - " + exceptionType.ToString());
            }

            return storedProcName;
        }

		/// <summary>
		/// Generates the exception for QC
		/// </summary>
		/// <param name="reagentGuid"></param>
		/// <param name="rackGuid"></param>
		/// <param name="bloodUnitGuid"></param>
		/// <param name="patientGuid"></param>
		/// <param name="previousResult"></param>
		/// <param name="currentResult"></param>
		/// <param name="phase"></param> 
		/// <param name="testingTechName"></param>
		public void GenerateExceptionQC
		(
			Guid reagentGuid, 
			Guid rackGuid, 
			Guid bloodUnitGuid, 
			Guid patientGuid, 
			string previousResult, 
			string currentResult, 
			string phase, 
			string testingTechName
		)
		{
			DataTable dtReagentData = DAL.Reagent.GetReagentByGuid(reagentGuid);
			string RackName = "";
			//
			DataTable dtRacks = BOL.Rack.GetDailyQCRacks(this.ExceptionDate);
			//
			if ((dtReagentData.Rows.Count == 0) ||
				(dtRacks.Rows.Count == 0))
			{
				this.ExceptionData = null;
				return;
			}
			//
			if (dtRacks.Select(TABLE.Rack.RackGuid + " = '" + rackGuid.ToString() + "'").Length > 0)
			{
				RackName = dtRacks.Select(TABLE.Rack.RackGuid + " = '" + rackGuid.ToString() + "'")[0][TABLE.Rack.RackName].ToString();
			}
			//
			DataTable dtExceptionQC = new DataTable(TABLE.ExceptionQc.TableName);
			dtExceptionQC.Columns.Add(TABLE.ExceptionQc.ExceptionQcGuid, typeof(Guid));
			dtExceptionQC.Columns.Add(TABLE.ExceptionQc.ExceptionTypeId, typeof(int));
			dtExceptionQC.Columns.Add(TABLE.ExceptionQc.ReagentName);
			dtExceptionQC.Columns.Add(TABLE.ExceptionQc.LotNumber);
			dtExceptionQC.Columns.Add(TABLE.ExceptionQc.LotExpirationDate, typeof(DateTime));
			dtExceptionQC.Columns.Add(TABLE.ExceptionQc.ReagentManufacturerName);
			dtExceptionQC.Columns.Add(TABLE.ExceptionQc.ReagentTypeName);
			dtExceptionQC.Columns.Add(TABLE.ExceptionQc.ExceptionComment);
			dtExceptionQC.Columns.Add(TABLE.ExceptionQc.ExceptionUser);
			dtExceptionQC.Columns.Add(TABLE.ExceptionQc.ExceptionDate, typeof(DateTime));
			dtExceptionQC.Columns.Add(TABLE.ExceptionQc.RackName);
			dtExceptionQC.Columns.Add(TABLE.ExceptionQc.PreviousResult);
			dtExceptionQC.Columns.Add(TABLE.ExceptionQc.CurrentResult);
			dtExceptionQC.Columns.Add(TABLE.ExceptionQc.ReactivityPhase);
			dtExceptionQC.Columns.Add(TABLE.ExceptionQc.BloodUnitGuid);
			dtExceptionQC.Columns.Add(TABLE.ExceptionQc.PatientGuid);
			dtExceptionQC.Columns.Add(TABLE.ExceptionQc.TestingTechName);
			dtExceptionQC.Columns.Add(TABLE.ExceptionQc.DivisionCode);
			//
			this.ExceptionData = dtExceptionQC.NewRow();
			this.ExceptionData[TABLE.ExceptionQc.ExceptionQcGuid] = Guid.NewGuid();
			this.ExceptionData[TABLE.ExceptionQc.ExceptionTypeId] = (int)this.ExceptionType;
			this.ExceptionData[TABLE.ExceptionQc.ReagentName] = dtReagentData.Rows[0][TABLE.Reagent.ReagentName];
			this.ExceptionData[TABLE.ExceptionQc.LotNumber] = dtReagentData.Rows[0][TABLE.Reagent.LotNumber];
			this.ExceptionData[TABLE.ExceptionQc.LotExpirationDate] = dtReagentData.Rows[0][TABLE.Reagent.LotExpirationDate];
			this.ExceptionData[TABLE.ExceptionQc.ReagentManufacturerName] = dtReagentData.Rows[0][TABLE.Reagent.ReagentManufacturerName];
			this.ExceptionData[TABLE.ExceptionQc.ReagentTypeName] = dtReagentData.Rows[0][TABLE.ReagentType.ReagentTypeName];
			this.ExceptionData[TABLE.ExceptionQc.ExceptionComment] = this.ExceptionComment;
			this.ExceptionData[TABLE.ExceptionQc.ExceptionUser] = this.ExceptionUser;
			this.ExceptionData[TABLE.ExceptionQc.ExceptionDate] = this.ExceptionDate;
			this.ExceptionData[TABLE.ExceptionQc.RackName] = RackName;
			this.ExceptionData[TABLE.ExceptionQc.PreviousResult] = previousResult;
			this.ExceptionData[TABLE.ExceptionQc.CurrentResult] = currentResult;
			this.ExceptionData[TABLE.ExceptionQc.ReactivityPhase] = phase;
			//
			if(!bloodUnitGuid.Equals(Guid.Empty))
			{
				this.ExceptionData[TABLE.ExceptionQc.BloodUnitGuid] = bloodUnitGuid;
			}
			//
			if(!patientGuid.Equals(Guid.Empty))
			{
				this.ExceptionData[TABLE.ExceptionQc.PatientGuid] = patientGuid;
			}
			//
			if (testingTechName == "")
			{
				this.ExceptionData[TABLE.ExceptionQc.TestingTechName] = this.ExceptionUser;
			}
			else
			{
				this.ExceptionData[TABLE.ExceptionQc.TestingTechName] = GetUserNameFromID(testingTechName);
			}
			this.ExceptionData[TABLE.ExceptionQc.DivisionCode] = Common.LogonUser.LogonUserDivisionCode;

			dtExceptionQC.Rows.Add(this.ExceptionData);
		}

		///<Developers>
		///	<Developer>Krzysztof Dobranowski</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/7/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4251"> 
		///		<ExpectedInput>Valid exception data</ExpectedInput>
		///		<ExpectedOutput>Valid exception DataRow</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4252"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Generates the exception for QC Inactivation
		/// </summary>
		/// <param name="rackName"></param>
		/// <param name="dtQcResults"></param>
		/// <param name="testingTech"></param>
		/// <returns></returns>
		public DataTable GenerateExceptionQCInactivation(string rackName, DataTable dtQcResults, string testingTech)
		{
			DataTable dtExceptionQCInactivation = new DataTable(TABLE.ExceptionQcInactivation.TableName);
			dtExceptionQCInactivation.Columns.Add(TABLE.ExceptionQcInactivation.ExceptionQcInactivationGuid, typeof(Guid));
			dtExceptionQCInactivation.Columns.Add(TABLE.ExceptionQcInactivation.ExceptionTypeId, typeof(int));
			dtExceptionQCInactivation.Columns.Add(TABLE.ExceptionQcInactivation.ExceptionComment);
			dtExceptionQCInactivation.Columns.Add(TABLE.ExceptionQcInactivation.ExceptionUser);
			dtExceptionQCInactivation.Columns.Add(TABLE.ExceptionQcInactivation.ExceptionDate, typeof(DateTime));
			dtExceptionQCInactivation.Columns.Add(TABLE.ExceptionQcInactivation.ReagentRack);
			dtExceptionQCInactivation.Columns.Add(TABLE.ExceptionQcInactivation.TestingTechName);
			dtExceptionQCInactivation.Columns.Add(TABLE.ExceptionQcInactivation.DivisionCode);

			this.ExceptionData = dtExceptionQCInactivation.NewRow();
			this.ExceptionData[TABLE.ExceptionQcInactivation.ExceptionQcInactivationGuid] = Guid.NewGuid();
			this.ExceptionData[TABLE.ExceptionQcInactivation.ExceptionTypeId] = (int)this.ExceptionType;
			this.ExceptionData[TABLE.ExceptionQcInactivation.ExceptionComment] = this.ExceptionComment;
			this.ExceptionData[TABLE.ExceptionQcInactivation.ExceptionUser] = this.GetUserNameFromID(this.ExceptionUser);
			this.ExceptionData[TABLE.ExceptionQcInactivation.ExceptionDate] = this.ExceptionDate;
			this.ExceptionData[TABLE.ExceptionQcInactivation.ReagentRack] = rackName;
			this.ExceptionData[TABLE.ExceptionQcInactivation.TestingTechName] = testingTech;
			this.ExceptionData[TABLE.ExceptionQcInactivation.DivisionCode] = Common.LogonUser.LogonUserDivisionCode;
			
			dtExceptionQCInactivation.Rows.Add(this.ExceptionData);

			DataTable dtExceptionQcInactivationReagents = new DataTable(TABLE.ExceptionQcInactivationReagent.TableName);
			dtExceptionQcInactivationReagents.Columns.Add(TABLE.ExceptionQcInactivationReagent.ExceptionQcInactivationReagentGuid, typeof(Guid));
			dtExceptionQcInactivationReagents.Columns.Add(TABLE.ExceptionQcInactivationReagent.ExceptionQcInactivationGuid, typeof(Guid));
			dtExceptionQcInactivationReagents.Columns.Add(TABLE.ExceptionQcInactivationReagent.ReagentTypeName);
			dtExceptionQcInactivationReagents.Columns.Add(TABLE.ExceptionQcInactivationReagent.TestWithName);
			dtExceptionQcInactivationReagents.Columns.Add(TABLE.ExceptionQcInactivationReagent.IsResult);
			dtExceptionQcInactivationReagents.Columns.Add(TABLE.ExceptionQcInactivationReagent.IatResult);
			dtExceptionQcInactivationReagents.Columns.Add(TABLE.ExceptionQcInactivationReagent.CcResult);

			foreach(DataRow drQcResult in dtQcResults.Rows)
			{
				if (drQcResult.IsNull(TABLE.RackLot.TestWithName) == true ||
					drQcResult.IsNull(ARTIFICIAL.IsResults) == true ||
					drQcResult.IsNull(ARTIFICIAL.IatResults) == true ||
					drQcResult.IsNull(ARTIFICIAL.CcResults) == true)
					continue;

				if (drQcResult[TABLE.RackLot.TestWithName].ToString().Trim() != "" &&
					(drQcResult[ARTIFICIAL.IsResults].ToString().Trim() != "" ||
					drQcResult[ARTIFICIAL.IatResults].ToString().Trim() != "" ||
					drQcResult[ARTIFICIAL.CcResults].ToString().Trim() != ""))
				{
					DataRow drExceptionQcInactivationReagent = dtExceptionQcInactivationReagents.NewRow();
					drExceptionQcInactivationReagent[TABLE.ExceptionQcInactivationReagent.ExceptionQcInactivationReagentGuid] = Guid.NewGuid();
					drExceptionQcInactivationReagent[TABLE.ExceptionQcInactivationReagent.ExceptionQcInactivationGuid] = this.ExceptionData[TABLE.ExceptionQcInactivation.ExceptionQcInactivationGuid];
					drExceptionQcInactivationReagent[TABLE.ExceptionQcInactivationReagent.ReagentTypeName] = drQcResult[TABLE.ReagentType.ReagentTypeName];
					drExceptionQcInactivationReagent[TABLE.ExceptionQcInactivationReagent.TestWithName] = drQcResult[TABLE.RackLot.TestWithName];
					drExceptionQcInactivationReagent[TABLE.ExceptionQcInactivationReagent.IsResult] = drQcResult[ARTIFICIAL.IsResults];
					drExceptionQcInactivationReagent[TABLE.ExceptionQcInactivationReagent.IatResult] = drQcResult[ARTIFICIAL.IatResults];
					drExceptionQcInactivationReagent[TABLE.ExceptionQcInactivationReagent.CcResult] = drQcResult[ARTIFICIAL.CcResults];

					dtExceptionQcInactivationReagents.Rows.Add(drExceptionQcInactivationReagent);
				}
			}

			return dtExceptionQcInactivationReagents;
		}

		///<Developers>
		///	<Developer>Krzysztof Dobranowski</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/7/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4253"> 
		///		<ExpectedInput>Valid exception data</ExpectedInput>
		///		<ExpectedOutput>Valid exception DataRow</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4254"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Generates exception when QC was not performed for a rack used for patient testing
		/// </summary>
		/// <param name="patient"></param>
		/// <param name="specimen"></param>
		/// <param name="rackName"></param>
		/// <param name="testingTechName"></param>
		public void GenerateExceptionQCNotPerformed(BOL.Patient patient, BOL.Specimen specimen, string rackName, string testingTechName)
		{
			DataTable dtExceptionQCNotPerformed = new DataTable(TABLE.ExceptionQcNotPerformed.TableName);
			dtExceptionQCNotPerformed.Columns.Add(TABLE.ExceptionQcNotPerformed.ExceptionQcNotPerformedGuid, typeof(Guid));
			dtExceptionQCNotPerformed.Columns.Add(TABLE.ExceptionQcNotPerformed.ExceptionTypeId, typeof(int));
			dtExceptionQCNotPerformed.Columns.Add(TABLE.ExceptionQcNotPerformed.PatientGuid, typeof(Guid));
			dtExceptionQCNotPerformed.Columns.Add(TABLE.ExceptionQcNotPerformed.PatientName);
			dtExceptionQCNotPerformed.Columns.Add(TABLE.ExceptionQcNotPerformed.PatientId);
			dtExceptionQCNotPerformed.Columns.Add(TABLE.ExceptionQcNotPerformed.SpecimenUid);
			dtExceptionQCNotPerformed.Columns.Add(TABLE.ExceptionQcNotPerformed.RackName);
			dtExceptionQCNotPerformed.Columns.Add(TABLE.ExceptionQcNotPerformed.TestingTechName);
			dtExceptionQCNotPerformed.Columns.Add(TABLE.ExceptionQcNotPerformed.ExceptionUser);
			dtExceptionQCNotPerformed.Columns.Add(TABLE.ExceptionQcNotPerformed.ExceptionComment);
			dtExceptionQCNotPerformed.Columns.Add(TABLE.ExceptionQcNotPerformed.ExceptionDate, typeof(DateTime));
			dtExceptionQCNotPerformed.Columns.Add(TABLE.ExceptionQcNotPerformed.DivisionCode);

			this.ExceptionData = dtExceptionQCNotPerformed.NewRow();
			this.ExceptionData[TABLE.ExceptionQcNotPerformed.ExceptionQcNotPerformedGuid] = Guid.NewGuid();
			this.ExceptionData[TABLE.ExceptionQcNotPerformed.ExceptionTypeId] = this.ExceptionType;
			this.ExceptionData[TABLE.ExceptionQcNotPerformed.PatientGuid] = patient.PatientGuid;
			this.ExceptionData[TABLE.ExceptionQcNotPerformed.PatientName] = patient.DisplayName;
			this.ExceptionData[TABLE.ExceptionQcNotPerformed.PatientId] = patient.DisplayPatientID;
			this.ExceptionData[TABLE.ExceptionQcNotPerformed.SpecimenUid] = specimen.SpecimenUid;
			this.ExceptionData[TABLE.ExceptionQcNotPerformed.RackName] = rackName;
			this.ExceptionData[TABLE.ExceptionQcNotPerformed.TestingTechName] = GetUserNameFromID(testingTechName);
			this.ExceptionData[TABLE.ExceptionQcNotPerformed.ExceptionUser] = this.ExceptionUser;
			this.ExceptionData[TABLE.ExceptionQcNotPerformed.ExceptionComment] = this.ExceptionComment;
			this.ExceptionData[TABLE.ExceptionQcNotPerformed.ExceptionDate] = this.ExceptionDate;
			this.ExceptionData[TABLE.ExceptionQcNotPerformed.DivisionCode] = Common.LogonUser.LogonUserDivisionCode;
		}

		///<Developers>
		///	<Developer>Krzysztof Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>4/24/2005</CreationDate>
		///	<TestCases>
		///<Case type="0" testid ="7258"> 
		///		<ExpectedInput>Exception data parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="7259"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <param name="eyeReadableUnitId"></param>
		/// <param name="productCode"></param>
		/// <param name="rackName"></param>
		/// <param name="testingTechName"></param>
		/// <summary>
		/// Generates exception when QC was not performed for a rack used for Unit Testing
		/// </summary>
		public void GenerateExceptionQCNotPerformed(string eyeReadableUnitId, string productCode, string rackName, string testingTechName)
		{
			DataTable dtExceptionQCNotPerformed = new DataTable(TABLE.ExceptionQcNotPerformed.TableName);
			dtExceptionQCNotPerformed.Columns.Add(TABLE.ExceptionQcNotPerformed.ExceptionQcNotPerformedGuid, typeof(Guid));
			dtExceptionQCNotPerformed.Columns.Add(TABLE.ExceptionQcNotPerformed.ExceptionTypeId, typeof(int));
			dtExceptionQCNotPerformed.Columns.Add(TABLE.ExceptionQcNotPerformed.UnitId, typeof(string));
			dtExceptionQCNotPerformed.Columns.Add(TABLE.ExceptionQcNotPerformed.ProductCode, typeof(string));
			dtExceptionQCNotPerformed.Columns.Add(TABLE.ExceptionQcNotPerformed.RackName);
			dtExceptionQCNotPerformed.Columns.Add(TABLE.ExceptionQcNotPerformed.TestingTechName);
			dtExceptionQCNotPerformed.Columns.Add(TABLE.ExceptionQcNotPerformed.ExceptionUser);
			dtExceptionQCNotPerformed.Columns.Add(TABLE.ExceptionQcNotPerformed.ExceptionComment);
			dtExceptionQCNotPerformed.Columns.Add(TABLE.ExceptionQcNotPerformed.ExceptionDate, typeof(DateTime));
			dtExceptionQCNotPerformed.Columns.Add(TABLE.ExceptionQcNotPerformed.DivisionCode);

			this.ExceptionData = dtExceptionQCNotPerformed.NewRow();
			this.ExceptionData[TABLE.ExceptionQcNotPerformed.ExceptionQcNotPerformedGuid] = Guid.NewGuid();
			this.ExceptionData[TABLE.ExceptionQcNotPerformed.ExceptionTypeId] = this.ExceptionType;
			this.ExceptionData[TABLE.ExceptionQcNotPerformed.UnitId] = eyeReadableUnitId;
			this.ExceptionData[TABLE.ExceptionQcNotPerformed.ProductCode] = productCode;
			this.ExceptionData[TABLE.ExceptionQcNotPerformed.RackName] = rackName;
			this.ExceptionData[TABLE.ExceptionQcNotPerformed.TestingTechName] = GetUserNameFromID(testingTechName);
			this.ExceptionData[TABLE.ExceptionQcNotPerformed.ExceptionUser] = this.ExceptionUser;
			this.ExceptionData[TABLE.ExceptionQcNotPerformed.ExceptionComment] = this.ExceptionComment;
			this.ExceptionData[TABLE.ExceptionQcNotPerformed.ExceptionDate] = this.ExceptionDate;
			this.ExceptionData[TABLE.ExceptionQcNotPerformed.DivisionCode] = Common.LogonUser.LogonUserDivisionCode;
		}

		///<Developers>
		///	<Developer>C. Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>10/26/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5886"> 
		///		<ExpectedInput>Patient, Specimen, BloodUnit</ExpectedInput>
		///		<ExpectedOutput>ExceptionData property has values</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5887"> 
		///		<ExpectedInput>Invalid Patient</ExpectedInput>
		///		<ExpectedOutput>ArgumentException</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5900"> 
		///		<ExpectedInput>Invalid BloodUnit</ExpectedInput>
		///		<ExpectedOutput>ArgumentException</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// GenerateExceptionCautionTagBTRF
		/// </summary>
		/// <param name="patient"></param>
		/// <param name="specimen"></param>
		/// <param name="bloodUnit"></param>
		public void GenerateExceptionCautionTagBTRF(BOL.Patient patient, BOL.Specimen specimen, BOL.BloodUnit bloodUnit)
		{
			if (patient == null)
			{
				throw new ArgumentException(Common.StrRes.SysErrMsg.Common.InvalidArgument("patient").ResString);
			}
			if (bloodUnit == null)
			{
				throw new ArgumentException(Common.StrRes.SysErrMsg.Common.InvalidArgument("blood unit").ResString);
			}

			DataTable dtExceptionCautionTagBTRF = new DataTable(TABLE.ExceptionCautionTag.TableName);
			dtExceptionCautionTagBTRF.Columns.Add(TABLE.ExceptionCautionTag.ExceptionCautionTagGuid, typeof(Guid));
			dtExceptionCautionTagBTRF.Columns.Add(TABLE.ExceptionCautionTag.ExceptionTypeId, typeof(int));
			dtExceptionCautionTagBTRF.Columns.Add(TABLE.ExceptionCautionTag.SpecimenUid);
			dtExceptionCautionTagBTRF.Columns.Add(TABLE.ExceptionCautionTag.PatientName);
			dtExceptionCautionTagBTRF.Columns.Add(TABLE.ExceptionCautionTag.PatientId);
			dtExceptionCautionTagBTRF.Columns.Add(TABLE.ExceptionCautionTag.PatientGuid, typeof(Guid));
			dtExceptionCautionTagBTRF.Columns.Add(TABLE.ExceptionCautionTag.PatientAboRh);
			dtExceptionCautionTagBTRF.Columns.Add(TABLE.ExceptionCautionTag.UnitId);
			dtExceptionCautionTagBTRF.Columns.Add(TABLE.ExceptionCautionTag.UnitGuid, typeof(Guid));
			dtExceptionCautionTagBTRF.Columns.Add(TABLE.ExceptionCautionTag.ProductCode);
			dtExceptionCautionTagBTRF.Columns.Add(TABLE.ExceptionCautionTag.UnitAboRh);
			dtExceptionCautionTagBTRF.Columns.Add(TABLE.ExceptionCautionTag.UnitExpirationDate, typeof(DateTime));
			dtExceptionCautionTagBTRF.Columns.Add(TABLE.ExceptionCautionTag.ExceptionUser);
			dtExceptionCautionTagBTRF.Columns.Add(TABLE.ExceptionCautionTag.ExceptionComment);
			dtExceptionCautionTagBTRF.Columns.Add(TABLE.ExceptionCautionTag.ExceptionDate, typeof(DateTime));
			dtExceptionCautionTagBTRF.Columns.Add(TABLE.ExceptionCautionTag.DivisionCode);
			dtExceptionCautionTagBTRF.Columns.Add(TABLE.ExceptionCautionTag.LastUpdateUser);

			this.ExceptionData = dtExceptionCautionTagBTRF.NewRow();
			this.ExceptionData[TABLE.ExceptionCautionTag.ExceptionCautionTagGuid] = Guid.NewGuid();
			this.ExceptionData[TABLE.ExceptionCautionTag.ExceptionTypeId] = this.ExceptionType;
			if (specimen != null) this.ExceptionData[TABLE.ExceptionCautionTag.SpecimenUid]		= specimen.SpecimenUid;
			this.ExceptionData[TABLE.ExceptionCautionTag.PatientName]		= patient.DisplayName;
			this.ExceptionData[TABLE.ExceptionCautionTag.PatientId]		= patient.DisplayPatientID;
			this.ExceptionData[TABLE.ExceptionCautionTag.PatientGuid]		= patient.PatientGuid;
			this.ExceptionData[TABLE.ExceptionCautionTag.PatientAboRh]	= patient.AboRh.AboRHText;
			this.ExceptionData[TABLE.ExceptionCautionTag.UnitId]			= bloodUnit.EyeReadableUnitId;
			this.ExceptionData[TABLE.ExceptionCautionTag.UnitGuid]		= bloodUnit.BloodUnitGuid;
			this.ExceptionData[TABLE.ExceptionCautionTag.ProductCode]		= bloodUnit.ProductCode;
			this.ExceptionData[TABLE.ExceptionCautionTag.UnitAboRh]		= bloodUnit.BloodUnitMedia.AboRh.AboRHText;
			this.ExceptionData[TABLE.ExceptionCautionTag.UnitExpirationDate] = bloodUnit.BloodUnitMedia.UnitExpirationDate;
			this.ExceptionData[TABLE.ExceptionCautionTag.ExceptionUser]	= this.ExceptionUser;
			this.ExceptionData[TABLE.ExceptionCautionTag.ExceptionComment] = this.ExceptionComment;
			this.ExceptionData[TABLE.ExceptionCautionTag.ExceptionDate]	= this.ExceptionDate;
			this.ExceptionData[TABLE.ExceptionCautionTag.DivisionCode]	= Common.LogonUser.LogonUserDivisionCode;
			this.ExceptionData[TABLE.ExceptionCautionTag.LastUpdateUser]	= Common.LogonUser.LogonUserName;
		}


		///<Developers>
		///	<Developer>C. Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/25/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6512"> 
		///		<ExpectedInput>Valid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="6513"> 
		///		<ExpectedInput>Invalid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// GenerateExceptionUnitIssue
		/// </summary>
		/// <param name="patientUnitIssue"></param>
		/// <param name="bloodUnit"></param>
		public void GenerateExceptionUnitIssue(BOL.PatientUnitIssue patientUnitIssue, BOL.BloodUnit bloodUnit)
		{
			if (patientUnitIssue == null)
			{
				throw new ArgumentException(Common.StrRes.SysErrMsg.Common.InvalidArgument("patient").ResString);
			}
			if (bloodUnit == null)
			{
				throw new ArgumentException(Common.StrRes.SysErrMsg.Common.InvalidArgument("blood unit").ResString);
			}
			
			BOL.Specimen specimen = patientUnitIssue.GetSpecimenAssociatedWithUnit(bloodUnit.BloodUnitGuid);

			string patientAntibodies = string.Empty;
			string componentTRs = string.Empty;
			string patientAntigenNeg = string.Empty;
			string unitAntigens = string.Empty;
			string techInterpretation = string.Empty;

			if (specimen != null)
			{
				DataRow [] dtRowAbo	= specimen.TestResults.Select(TABLE.BloodUnitTest.BloodTestTypeId + " = " + (int) Common.TestType.ABOInterpTAS);
				DataRow [] dtRowRh = specimen.TestResults.Select(TABLE.BloodUnitTest.BloodTestTypeId + " = " + (int) Common.TestType.RhInterpTAS);

				if (dtRowAbo.Length > 0)
				{
					if (dtRowRh.Length > 0)
					{
						string rh = dtRowRh[0][TABLE.SpecimenTest.TestResultId].ToString();
						if (rh.Trim() == "P") rh = "Pos";
						else if (rh.Trim() == "N") rh = "Neg";
						techInterpretation = dtRowAbo[0][TABLE.SpecimenTest.TestResultId].ToString().Trim() + " " + rh;
							
					}
					else
					{
						techInterpretation = dtRowAbo[0][TABLE.SpecimenTest.TestResultId].ToString();
					}
				}
			}

			//CR 2910 Added conversion antibodies
			DataRow[] drPatientTRs = patientUnitIssue.Patient.TransfusionRequirements.Select(TABLE.PatientTransfusionRequirement.RecordStatusCode + " = 'A' OR " + TABLE.PatientTransfusionRequirement.RecordStatusCode + " = 'C'", TABLE.PatientTransfusionRequirement.TransfusionRequirementCategoryCode);

			foreach(DataRow drPatientTR in drPatientTRs)
			{
				//antibodies identified
				if (drPatientTR[TABLE.PatientTransfusionRequirement.TransfusionRequirementCategoryCode].ToString().Trim() == Common.Utility.GetTransfusionCategoryCodeFromEnum(Common.TransfusionRequirementCategoryCode.AntibodyIdentified))
				{
					if (patientAntibodies != "")
					{
						patientAntibodies += ", ";
					}
					patientAntibodies += drPatientTR[TABLE.PatientTransfusionRequirement.TransfusionRequirementText].ToString().Trim();
				}
				if (drPatientTR[TABLE.PatientTransfusionRequirement.TransfusionRequirementCategoryCode].ToString().Trim() == Common.Utility.GetTransfusionCategoryCodeFromEnum(Common.TransfusionRequirementCategoryCode.AntigenNegative))
				{
					if (patientAntigenNeg != "")
					{
						patientAntigenNeg += ", ";
					}
					patientAntigenNeg += drPatientTR[TABLE.PatientTransfusionRequirement.TransfusionRequirementText].ToString().Trim();
				}
					//antigen negative and component TRs
				else if (drPatientTR[TABLE.PatientTransfusionRequirement.TransfusionRequirementCategoryCode].ToString().Trim() == Common.Utility.GetTransfusionCategoryCodeFromEnum(Common.TransfusionRequirementCategoryCode.ComponentRequirement))
				{
					if (componentTRs != "")
					{
						componentTRs += ", ";
					}
					componentTRs += drPatientTR[TABLE.PatientTransfusionRequirement.TransfusionRequirementText].ToString().Trim();
				}
			}
			
			DataTable dtBloodUnitAntigens = BOL.BloodUnitAntigen.GetBloodUnitAntigens(bloodUnit.BloodUnitGuid, Common.LogonUser.LogonUserDivisionCode);
			foreach(DataRow drBloodUnitAntigen in dtBloodUnitAntigens.Rows)
			{
				if (unitAntigens != "")
				{
					unitAntigens += ", ";
				}

				unitAntigens += drBloodUnitAntigen[TABLE.AntigenType.AntigenTypeName].ToString();
				if ((bool)drBloodUnitAntigen[TABLE.BloodUnitAntigen.Positive] == true)
				{
					unitAntigens += " Pos";
				}
				else
				{
					unitAntigens += " Neg";
				}
			}

			ArrayList bloodUnitAntigenTypingTests = BOL.BloodUnitTest.GetAntigenTypingTests(bloodUnit.BloodUnitGuid, Common.LogonUser.LogonUserDivisionCode);
			foreach(BOL.BloodUnitTest bloodUnitTest in bloodUnitAntigenTypingTests)
			{
				if (bloodUnitTest.BloodTestTypeObj.BloodTestName.IndexOf(ARTIFICIAL.Interp) != -1)
				{
					if (unitAntigens != "")
					{
						unitAntigens += ", ";
					}

					unitAntigens += bloodUnitTest.Specificity;
					if (bloodUnitTest.TestResultId.Trim() == "P")
					{
						unitAntigens += " Pos";
					}
					else if (bloodUnitTest.TestResultId.Trim() == "N")
					{
						unitAntigens += " Neg";
					}
					else if (bloodUnitTest.TestResultId.Trim() == "I")
					{
						unitAntigens += " Inc";
					}
				}
			}

			DataTable dtExceptionUnitIssue = new DataTable(TABLE.ExceptionUnitIssue.TableName);
			dtExceptionUnitIssue.Columns.Add(TABLE.ExceptionUnitIssue.ExceptionUnitIssueGuid, typeof(Guid));
			dtExceptionUnitIssue.Columns.Add(TABLE.ExceptionUnitIssue.ExceptionTypeId, typeof(int));
			dtExceptionUnitIssue.Columns.Add(TABLE.ExceptionUnitIssue.PatientName);
			dtExceptionUnitIssue.Columns.Add(TABLE.ExceptionUnitIssue.PatientId);
			dtExceptionUnitIssue.Columns.Add(TABLE.ExceptionUnitIssue.PatientGuid, typeof(Guid));
			dtExceptionUnitIssue.Columns.Add(TABLE.ExceptionUnitIssue.HistoricAboRh);
			dtExceptionUnitIssue.Columns.Add(TABLE.ExceptionUnitIssue.TechAboRh);
			dtExceptionUnitIssue.Columns.Add(TABLE.ExceptionUnitIssue.SystemAboRh);
			dtExceptionUnitIssue.Columns.Add(TABLE.ExceptionUnitIssue.PatientAntibodies);
			dtExceptionUnitIssue.Columns.Add(TABLE.ExceptionUnitIssue.PatientTransfusionRequirements);
			dtExceptionUnitIssue.Columns.Add(TABLE.ExceptionUnitIssue.SpecimenUid);
			dtExceptionUnitIssue.Columns.Add(TABLE.ExceptionUnitIssue.UnitId);
			dtExceptionUnitIssue.Columns.Add(TABLE.ExceptionUnitIssue.UnitGuid, typeof(Guid));
			dtExceptionUnitIssue.Columns.Add(TABLE.ExceptionUnitIssue.UnitProductType);
			dtExceptionUnitIssue.Columns.Add(TABLE.ExceptionUnitIssue.UnitAboRh);
			dtExceptionUnitIssue.Columns.Add(TABLE.ExceptionUnitIssue.UnitExpirationDate, typeof(DateTime));
			dtExceptionUnitIssue.Columns.Add(TABLE.ExceptionUnitIssue.UnitAntigens);
			dtExceptionUnitIssue.Columns.Add(TABLE.ExceptionUnitIssue.UnitDonationType);
			dtExceptionUnitIssue.Columns.Add(TABLE.ExceptionUnitIssue.UnitTestingEntry);
			dtExceptionUnitIssue.Columns.Add(TABLE.ExceptionUnitIssue.OrderIndicator);
			dtExceptionUnitIssue.Columns.Add(TABLE.ExceptionUnitIssue.OrderedBy);
			dtExceptionUnitIssue.Columns.Add(TABLE.ExceptionUnitIssue.IssuedToLocation);
			dtExceptionUnitIssue.Columns.Add(TABLE.ExceptionUnitIssue.TransporterName);
			dtExceptionUnitIssue.Columns.Add(TABLE.ExceptionUnitIssue.IssuingTechName);
			dtExceptionUnitIssue.Columns.Add(TABLE.ExceptionUnitIssue.LoginTechName);
			dtExceptionUnitIssue.Columns.Add(TABLE.ExceptionUnitIssue.IssueDateTime, typeof(DateTime));
			dtExceptionUnitIssue.Columns.Add(TABLE.ExceptionUnitIssue.ExceptionUser);
			dtExceptionUnitIssue.Columns.Add(TABLE.ExceptionUnitIssue.ExceptionComment);
			dtExceptionUnitIssue.Columns.Add(TABLE.ExceptionUnitIssue.ExceptionDate, typeof(DateTime));
			dtExceptionUnitIssue.Columns.Add(TABLE.ExceptionUnitIssue.DivisionCode);
			//dtExceptionUnitIssue.Columns.Add(TABLE.ExceptionUnitIssue.LastUpdateDate);
			dtExceptionUnitIssue.Columns.Add(TABLE.ExceptionUnitIssue.LastUpdateUser);

			this.ExceptionData = dtExceptionUnitIssue.NewRow();
			this.ExceptionData[TABLE.ExceptionUnitIssue.ExceptionUnitIssueGuid] = Guid.NewGuid();
			this.ExceptionData[TABLE.ExceptionUnitIssue.ExceptionTypeId]	= this.ExceptionType;
			this.ExceptionData[TABLE.ExceptionUnitIssue.PatientName]		= patientUnitIssue.Patient.DisplayName;
			this.ExceptionData[TABLE.ExceptionUnitIssue.PatientId]		= patientUnitIssue.Patient.DisplayPatientID;
			this.ExceptionData[TABLE.ExceptionUnitIssue.PatientGuid]		= patientUnitIssue.Patient.PatientGuid;
			this.ExceptionData[TABLE.ExceptionUnitIssue.HistoricAboRh]	= patientUnitIssue.Patient.AboRh.AboRHText;
			this.ExceptionData[TABLE.ExceptionUnitIssue.TechAboRh]		= techInterpretation;
			// System interpretation, ToDo
			string aboSystem = patientUnitIssue.GetSystemAbo();
            //CR 3289
			string rhSystem = patientUnitIssue.GetSystemRh();

            if (aboSystem.Trim() == string.Empty || rhSystem.Trim() == string.Empty)
            {
                this.ExceptionData[TABLE.ExceptionUnitIssue.SystemAboRh] = "Not Available";
            }
            else
            {

                if (rhSystem.Trim() == "P") rhSystem = "Pos";
                else if (rhSystem.Trim() == "N") rhSystem = "Neg";
                this.ExceptionData[TABLE.ExceptionUnitIssue.SystemAboRh] = aboSystem + " " + rhSystem;
            }

		    this.ExceptionData[TABLE.ExceptionUnitIssue.PatientAntibodies]= patientAntibodies;
			this.ExceptionData[TABLE.ExceptionUnitIssue.PatientTransfusionRequirements] = componentTRs;
			if (specimen!=null)
			{
				this.ExceptionData[TABLE.ExceptionUnitIssue.SpecimenUid]	= specimen.SpecimenUid;
			}
			this.ExceptionData[TABLE.ExceptionUnitIssue.UnitId]			= bloodUnit.EyeReadableUnitId;
			this.ExceptionData[TABLE.ExceptionUnitIssue.UnitGuid]			= bloodUnit.BloodUnitGuid;
			this.ExceptionData[TABLE.ExceptionUnitIssue.UnitProductType]	= bloodUnit.ProductShortName;
			this.ExceptionData[TABLE.ExceptionUnitIssue.UnitAboRh]		= bloodUnit.BloodUnitMedia.AboRh.AboRHText;
			this.ExceptionData[TABLE.ExceptionUnitIssue.UnitExpirationDate] = bloodUnit.BloodUnitMedia.UnitExpirationDate;
			this.ExceptionData[TABLE.ExceptionUnitIssue.UnitAntigens]		= unitAntigens;
			this.ExceptionData[TABLE.ExceptionUnitIssue.UnitDonationType]	= bloodUnit.DonationType.DonationTypeText;
			string testingStatusText = "Unknown";
			if (bloodUnit.BloodUnitMedia.TestingStatusId > 0)
			{
				testingStatusText = DAL.TestingStatus.GetTestingStatusTextFromID(bloodUnit.BloodUnitMedia.TestingStatusId);
			}
			this.ExceptionData[TABLE.ExceptionUnitIssue.UnitTestingEntry]	= bloodUnit.BloodUnitMedia.TestingStatusId;
			this.ExceptionData[TABLE.ExceptionUnitIssue.OrderIndicator]	= patientUnitIssue.IsEmergencyIssue(bloodUnit.BloodUnitGuid) ? "Yes" : "No";

			this.ExceptionData[TABLE.ExceptionUnitIssue.OrderedBy]		= patientUnitIssue.OrderingPhysician;
			this.ExceptionData[TABLE.ExceptionUnitIssue.IssuedToLocation]	= patientUnitIssue.IssueToLocation;
			this.ExceptionData[TABLE.ExceptionUnitIssue.TransporterName]	= patientUnitIssue.Transporter;
			this.ExceptionData[TABLE.ExceptionUnitIssue.IssuingTechName]	= patientUnitIssue.IssuedBy;
			this.ExceptionData[TABLE.ExceptionUnitIssue.LoginTechName]	= Common.LogonUser.LogonUserName;
			this.ExceptionData[TABLE.ExceptionUnitIssue.IssueDateTime]	= patientUnitIssue.IssueDate;
			this.ExceptionData[TABLE.ExceptionUnitIssue.ExceptionUser]	= this.ExceptionUser;
			this.ExceptionData[TABLE.ExceptionUnitIssue.ExceptionComment]	= this.ExceptionComment;
			this.ExceptionData[TABLE.ExceptionUnitIssue.ExceptionDate]	= this.ExceptionDate;
			this.ExceptionData[TABLE.ExceptionUnitIssue.DivisionCode]	= Common.LogonUser.LogonUserDivisionCode;
			//this.ExceptionDate[TABLE.ExceptionUnitIssue.LastUpdateDate]	= BOL.VBECSDateTime.GetDivisionCurrentDateTime().ToString();
			this.ExceptionData[TABLE.ExceptionUnitIssue.LastUpdateUser]	= Common.LogonUser.LogonUserName;

			
			for (int i=0; i<ExceptionData.Table.Columns.Count; i++)
			{
				if (ExceptionData[i] == System.DBNull.Value)
				{
					ExceptionData[i] = string.Empty;
				}
			}
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/8/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="7457"> 
		///		<ExpectedInput>Valid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="7458"> 
		///		<ExpectedInput>Invalid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// GenerateExceptionEmergencyIssueOrder
		/// </summary>
		/// <param name="orderedComponent"></param>
		public void GenerateExceptionEmergencyIssueOrder(BOL.OrderedComponent orderedComponent)
		{
			//			if (patient == null)
			//			{
			//				throw new ArgumentException(Common.StrRes.SysErrMsg.Common.InvalidArgument("patient").ResString);
			//			}
			if (orderedComponent == null)
			{
				throw new ArgumentException(Common.StrRes.SysErrMsg.Common.InvalidArgument("orderedComponent").ResString);
			}
			//			if (patientOrder == null)
			//			{
			//				throw new ArgumentException(Common.StrRes.SysErrMsg.Common.InvalidArgument("patientOrder").ResString);
			//			}
			BOL.Patient patient = orderedComponent.Patient;
			BOL.PatientOrder patientOrder = (BOL.PatientOrder) orderedComponent;
			
			

			DataTable dtExceptionEmergencyIssueOrder = new DataTable(TABLE.ExceptionEmergencyIssueOrder.TableName);
			dtExceptionEmergencyIssueOrder.Columns.Add(TABLE.ExceptionEmergencyIssueOrder.ExceptionEmergencyIssueOrderGuid, typeof(Guid));
			dtExceptionEmergencyIssueOrder.Columns.Add(TABLE.ExceptionEmergencyIssueOrder.ExceptionTypeId, typeof(int));
			dtExceptionEmergencyIssueOrder.Columns.Add(TABLE.ExceptionEmergencyIssueOrder.PatientName);
			dtExceptionEmergencyIssueOrder.Columns.Add(TABLE.ExceptionEmergencyIssueOrder.PatientId);
			dtExceptionEmergencyIssueOrder.Columns.Add(TABLE.ExceptionEmergencyIssueOrder.PatientGuid, typeof(Guid));
			dtExceptionEmergencyIssueOrder.Columns.Add(TABLE.ExceptionEmergencyIssueOrder.CprsOrderNumber);
			dtExceptionEmergencyIssueOrder.Columns.Add(TABLE.ExceptionEmergencyIssueOrder.UrgencyIndicator);
			dtExceptionEmergencyIssueOrder.Columns.Add(TABLE.ExceptionEmergencyIssueOrder.ComponentClassOrdered);
			dtExceptionEmergencyIssueOrder.Columns.Add(TABLE.ExceptionEmergencyIssueOrder.DateWanted);
			dtExceptionEmergencyIssueOrder.Columns.Add(TABLE.ExceptionEmergencyIssueOrder.DateAcknowledged);
			dtExceptionEmergencyIssueOrder.Columns.Add(TABLE.ExceptionEmergencyIssueOrder.ProcessingTechName);
			dtExceptionEmergencyIssueOrder.Columns.Add(TABLE.ExceptionEmergencyIssueOrder.ExceptionUser);
			dtExceptionEmergencyIssueOrder.Columns.Add(TABLE.ExceptionEmergencyIssueOrder.ExceptionComment);
			dtExceptionEmergencyIssueOrder.Columns.Add(TABLE.ExceptionEmergencyIssueOrder.ExceptionDate);
			dtExceptionEmergencyIssueOrder.Columns.Add(TABLE.ExceptionEmergencyIssueOrder.DivisionCode);
			//dtExceptionEmergencyIssueOrder.Columns.Add(TABLE.ExceptionEmergencyIssueOrder.LastUpdateDate, typeof(DateTime));
			dtExceptionEmergencyIssueOrder.Columns.Add(TABLE.ExceptionEmergencyIssueOrder.LastUpdateUser);

			this.ExceptionData = dtExceptionEmergencyIssueOrder.NewRow();
			this.ExceptionData[TABLE.ExceptionEmergencyIssueOrder.ExceptionEmergencyIssueOrderGuid] = Guid.NewGuid();
			this.ExceptionData[TABLE.ExceptionEmergencyIssueOrder.ExceptionTypeId]	= this.ExceptionType;
			this.ExceptionData[TABLE.ExceptionEmergencyIssueOrder.PatientName]		= patient.DisplayName;
			this.ExceptionData[TABLE.ExceptionEmergencyIssueOrder.PatientId]			= patient.DisplayPatientID;
			this.ExceptionData[TABLE.ExceptionEmergencyIssueOrder.PatientGuid]		= patient.PatientGuid;
			this.ExceptionData[TABLE.ExceptionEmergencyIssueOrder.CprsOrderNumber]	= patientOrder.OrderGroupNumber;
			this.ExceptionData[TABLE.ExceptionEmergencyIssueOrder.UrgencyIndicator]	= orderedComponent.OrderUrgencyText;
			this.ExceptionData[TABLE.ExceptionEmergencyIssueOrder.ComponentClassOrdered] = orderedComponent.ComponentOrdered.BloodComponentName;
			this.ExceptionData[TABLE.ExceptionEmergencyIssueOrder.DateWanted]			= orderedComponent.RequiredDateTime;
			this.ExceptionData[TABLE.ExceptionEmergencyIssueOrder.DateAcknowledged]	= patientOrder.OrderReceivedDateTime;
			this.ExceptionData[TABLE.ExceptionEmergencyIssueOrder.ProcessingTechName]	= this.ExceptionUser;
			this.ExceptionData[TABLE.ExceptionUnitIssue.ExceptionUser]	= this.ExceptionUser;
			this.ExceptionData[TABLE.ExceptionUnitIssue.ExceptionComment]	= this.ExceptionComment;
			this.ExceptionData[TABLE.ExceptionUnitIssue.ExceptionDate]	= this.ExceptionDate;
			this.ExceptionData[TABLE.ExceptionUnitIssue.DivisionCode]	= Common.LogonUser.LogonUserDivisionCode;
			//this.ExceptionDate[TABLE.ExceptionUnitIssue.LastUpdateDate]	= BOL.VBECSDateTime.GetDivisionCurrentDateTime().ToString();
			this.ExceptionData[TABLE.ExceptionUnitIssue.LastUpdateUser]	= Common.LogonUser.LogonUserName;

			for (int i=0; i<ExceptionData.Table.Columns.Count; i++)
			{
				if (ExceptionData[i] == System.DBNull.Value)
				{
					ExceptionData[i] = string.Empty;
				}
			}
		}

		///<Developers>
		///	<Developer>Krzysztof Dobranowski</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/7/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4255"> 
		///		<ExpectedInput>Valid exception data</ExpectedInput>
		///		<ExpectedOutput>Valid exception DataRow</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4256"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Generates exception for patient AboRh discrepancy
		/// </summary>
		/// <param name="patient"></param>
		/// <param name="previousSpecimenUID"></param>
		/// <param name="previousSpecimenAbo"></param>
		/// <param name="previousSpecimenRh"></param>
		/// <param name="previousTestingTech"></param>
		/// <param name="previousEntryTech"></param>
		/// <param name="previousDivision"></param>
		/// <param name="previousTestDate"></param>
		/// <param name="currentSpecimenUID"></param>
		/// <param name="currentSpecimenAbo"></param>
		/// <param name="currentSpecimenRh"></param>
		/// <param name="currentTestingTech"></param>
		/// <param name="currentEntryTech"></param>
		/// <param name="currentDivision"></param>
		/// <param name="currentTestDate"></param>
		public void GenerateExceptionPatientAboRhDiscrepancy(BOL.Patient patient, string previousSpecimenUID, string previousSpecimenAbo, string previousSpecimenRh, string previousTestingTech, string previousEntryTech, string previousDivision, DateTime previousTestDate,
			string currentSpecimenUID, string currentSpecimenAbo, string currentSpecimenRh, string currentTestingTech, string currentEntryTech, string currentDivision, DateTime currentTestDate)
		{
			string previousAboRh = CombineAboRh(previousSpecimenAbo, previousSpecimenRh);
			string currentAboRh = CombineAboRh(currentSpecimenAbo, currentSpecimenRh);
			BOL.Division pDivision, cDivision;

			try
			{
				pDivision = new BOL.Division(previousDivision);
			}
			catch (Common.DataAccessLayerException)
			{
				pDivision = null;
			}

			try
			{
				cDivision = new BOL.Division(currentDivision);
			}
			catch (Common.DataAccessLayerException)
			{
				cDivision = null;
			}

			DataTable dtPatientAboRhDiscrepancy = new DataTable(TABLE.ExceptionPatientAboRhDiscrepancy.TableName);
			dtPatientAboRhDiscrepancy.Columns.Add(TABLE.ExceptionPatientAboRhDiscrepancy.ExceptionPatientAboRhDiscrepancyGuid, typeof(Guid));
			dtPatientAboRhDiscrepancy.Columns.Add(TABLE.ExceptionPatientAboRhDiscrepancy.ExceptionTypeId, typeof(int));
			dtPatientAboRhDiscrepancy.Columns.Add(TABLE.ExceptionPatientAboRhDiscrepancy.PatientGuid, typeof(Guid));
			dtPatientAboRhDiscrepancy.Columns.Add(TABLE.ExceptionPatientAboRhDiscrepancy.PatientId);
			dtPatientAboRhDiscrepancy.Columns.Add(TABLE.ExceptionPatientAboRhDiscrepancy.PatientName);
			dtPatientAboRhDiscrepancy.Columns.Add(TABLE.ExceptionPatientAboRhDiscrepancy.PreviousSpecimenUid);
			dtPatientAboRhDiscrepancy.Columns.Add(TABLE.ExceptionPatientAboRhDiscrepancy.PreviousSpecimenAboRh);
			dtPatientAboRhDiscrepancy.Columns.Add(TABLE.ExceptionPatientAboRhDiscrepancy.PreviousTestingTechName);
			dtPatientAboRhDiscrepancy.Columns.Add(TABLE.ExceptionPatientAboRhDiscrepancy.PreviousEntryTechName);
			dtPatientAboRhDiscrepancy.Columns.Add(TABLE.ExceptionPatientAboRhDiscrepancy.PreviousTestingDivision);
			dtPatientAboRhDiscrepancy.Columns.Add(TABLE.ExceptionPatientAboRhDiscrepancy.PreviousTestingDate, typeof(DateTime));
			dtPatientAboRhDiscrepancy.Columns.Add(TABLE.ExceptionPatientAboRhDiscrepancy.CurrentSpecimenUid);
			dtPatientAboRhDiscrepancy.Columns.Add(TABLE.ExceptionPatientAboRhDiscrepancy.CurrentSpecimenAboRh);
			dtPatientAboRhDiscrepancy.Columns.Add(TABLE.ExceptionPatientAboRhDiscrepancy.CurrentTestingTechName);
			dtPatientAboRhDiscrepancy.Columns.Add(TABLE.ExceptionPatientAboRhDiscrepancy.CurrentEntryTechName);
			dtPatientAboRhDiscrepancy.Columns.Add(TABLE.ExceptionPatientAboRhDiscrepancy.CurrentTestingDivision);
			dtPatientAboRhDiscrepancy.Columns.Add(TABLE.ExceptionPatientAboRhDiscrepancy.CurrentTestingDate, typeof(DateTime));
			dtPatientAboRhDiscrepancy.Columns.Add(TABLE.ExceptionPatientAboRhDiscrepancy.ExceptionUser);
			dtPatientAboRhDiscrepancy.Columns.Add(TABLE.ExceptionPatientAboRhDiscrepancy.ExceptionComment);
			dtPatientAboRhDiscrepancy.Columns.Add(TABLE.ExceptionPatientAboRhDiscrepancy.DivisionCode);
			dtPatientAboRhDiscrepancy.Columns.Add(TABLE.ExceptionPatientAboRhDiscrepancy.ExceptionDate, typeof(DateTime));

			this.ExceptionData = dtPatientAboRhDiscrepancy.NewRow();
			this.ExceptionData[TABLE.ExceptionPatientAboRhDiscrepancy.ExceptionPatientAboRhDiscrepancyGuid] = Guid.NewGuid();
			this.ExceptionData[TABLE.ExceptionPatientAboRhDiscrepancy.ExceptionTypeId] = this.ExceptionType;
			this.ExceptionData[TABLE.ExceptionPatientAboRhDiscrepancy.PatientGuid] = patient.PatientGuid;
			this.ExceptionData[TABLE.ExceptionPatientAboRhDiscrepancy.PatientId] = patient.DisplayPatientID;
			this.ExceptionData[TABLE.ExceptionPatientAboRhDiscrepancy.PatientName] = patient.DisplayName;
			this.ExceptionData[TABLE.ExceptionPatientAboRhDiscrepancy.PreviousSpecimenUid] = previousSpecimenUID;
			this.ExceptionData[TABLE.ExceptionPatientAboRhDiscrepancy.PreviousSpecimenAboRh] = previousAboRh;
			this.ExceptionData[TABLE.ExceptionPatientAboRhDiscrepancy.PreviousTestingTechName] = GetUserNameFromID(previousTestingTech);
			this.ExceptionData[TABLE.ExceptionPatientAboRhDiscrepancy.PreviousEntryTechName] = GetUserNameFromID(previousEntryTech);
			if (pDivision != null)
				this.ExceptionData[TABLE.ExceptionPatientAboRhDiscrepancy.PreviousTestingDivision] = pDivision.DivisionName;
			else
				this.ExceptionData[TABLE.ExceptionPatientAboRhDiscrepancy.PreviousTestingDivision] = "";
			if (previousTestDate != DateTime.MinValue)
				this.ExceptionData[TABLE.ExceptionPatientAboRhDiscrepancy.PreviousTestingDate] = previousTestDate;
			else
				this.ExceptionData[TABLE.ExceptionPatientAboRhDiscrepancy.PreviousTestingDate] = DBNull.Value;
			this.ExceptionData[TABLE.ExceptionPatientAboRhDiscrepancy.CurrentSpecimenUid] = currentSpecimenUID;
			this.ExceptionData[TABLE.ExceptionPatientAboRhDiscrepancy.CurrentSpecimenAboRh] = currentAboRh;
			this.ExceptionData[TABLE.ExceptionPatientAboRhDiscrepancy.CurrentTestingTechName] = GetUserNameFromID(currentTestingTech);
			this.ExceptionData[TABLE.ExceptionPatientAboRhDiscrepancy.CurrentEntryTechName] = GetUserNameFromID(currentEntryTech);
			if (cDivision != null)
				this.ExceptionData[TABLE.ExceptionPatientAboRhDiscrepancy.CurrentTestingDivision] = cDivision.DivisionName;
			else
				this.ExceptionData[TABLE.ExceptionPatientAboRhDiscrepancy.CurrentTestingDivision] = "";
			this.ExceptionData[TABLE.ExceptionPatientAboRhDiscrepancy.CurrentTestingDate] = currentTestDate;
			this.ExceptionData[TABLE.ExceptionPatientAboRhDiscrepancy.ExceptionUser] = GetUserNameFromID(this.ExceptionUser);
			this.ExceptionData[TABLE.ExceptionPatientAboRhDiscrepancy.ExceptionComment] = this.ExceptionComment;
			this.ExceptionData[TABLE.ExceptionPatientAboRhDiscrepancy.DivisionCode] = Common.LogonUser.LogonUserDivisionCode;
			this.ExceptionData[TABLE.ExceptionPatientAboRhDiscrepancy.ExceptionDate] = this.ExceptionDate;
		}

		///<Developers>
		///	<Developer>Krzysztof Dobranowski</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/7/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4257"> 
		///		<ExpectedInput>Valid exception data</ExpectedInput>
		///		<ExpectedOutput>Valid exception DataRow</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4258"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Generates exception for reflex test ordered on expired specimen
		/// </summary>
		/// <param name="specimen"></param>
		/// <param name="ReflexOrderName"></param>
		public void GenerateExceptionReflexTest(BOL.Specimen specimen, string ReflexOrderName)
		{
			if (specimen.Patient == null)
			{
				this._exceptionData = null;
				return;
			}

			DataTable dtExceptionReflexTest = new DataTable(TABLE.ExceptionReflexTest.TableName);
			dtExceptionReflexTest.Columns.Add(TABLE.ExceptionReflexTest.ExceptionReflexTestGuid, typeof(Guid));
			dtExceptionReflexTest.Columns.Add(TABLE.ExceptionReflexTest.ExceptionTypeId, typeof(int));
			dtExceptionReflexTest.Columns.Add(TABLE.ExceptionReflexTest.PatientGuid, typeof(Guid));
			dtExceptionReflexTest.Columns.Add(TABLE.ExceptionReflexTest.PatientId);
			dtExceptionReflexTest.Columns.Add(TABLE.ExceptionReflexTest.PatientName);
			dtExceptionReflexTest.Columns.Add(TABLE.ExceptionReflexTest.SpecimenUid);
			dtExceptionReflexTest.Columns.Add(TABLE.ExceptionReflexTest.SpecimenExpirationDate, typeof(DateTime));
			dtExceptionReflexTest.Columns.Add(TABLE.ExceptionReflexTest.TestOrdered);
			dtExceptionReflexTest.Columns.Add(TABLE.ExceptionReflexTest.TestingTechName);
			dtExceptionReflexTest.Columns.Add(TABLE.ExceptionReflexTest.ExceptionUser);
			dtExceptionReflexTest.Columns.Add(TABLE.ExceptionReflexTest.ExceptionComment);
			dtExceptionReflexTest.Columns.Add(TABLE.ExceptionReflexTest.ExceptionDate, typeof(DateTime));
			dtExceptionReflexTest.Columns.Add(TABLE.ExceptionReflexTest.DivisionCode);

			this.ExceptionData = dtExceptionReflexTest.NewRow();
			this.ExceptionData[TABLE.ExceptionReflexTest.ExceptionReflexTestGuid] = Guid.NewGuid();
			this.ExceptionData[TABLE.ExceptionReflexTest.ExceptionTypeId] = this.ExceptionType;
			this.ExceptionData[TABLE.ExceptionReflexTest.PatientGuid] = specimen.Patient.PatientGuid;
			this.ExceptionData[TABLE.ExceptionReflexTest.PatientId] = specimen.Patient.DisplayPatientID;
			this.ExceptionData[TABLE.ExceptionReflexTest.PatientName] = specimen.Patient.DisplayName;
			this.ExceptionData[TABLE.ExceptionReflexTest.SpecimenUid] = specimen.SpecimenUid;
			this.ExceptionData[TABLE.ExceptionReflexTest.SpecimenExpirationDate] = specimen.SpecimenExpirationDate;
			this.ExceptionData[TABLE.ExceptionReflexTest.TestOrdered] = ReflexOrderName;
			this.ExceptionData[TABLE.ExceptionReflexTest.TestingTechName] = this.ExceptionUser;
			this.ExceptionData[TABLE.ExceptionReflexTest.ExceptionUser] = this.ExceptionData[TABLE.ExceptionReflexTest.TestingTechName];
			this.ExceptionData[TABLE.ExceptionReflexTest.ExceptionComment] = this.ExceptionComment;
			this.ExceptionData[TABLE.ExceptionReflexTest.ExceptionDate] = this.ExceptionDate;
			this.ExceptionData[TABLE.ExceptionReflexTest.DivisionCode] = Common.LogonUser.LogonUserDivisionCode;
		}

		///<Developers>
		///	<Developer>Krzysztof Dobranowski</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/7/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4259"> 
		///		<ExpectedInput>Valid exception data</ExpectedInput>
		///		<ExpectedOutput>Valid exception DataRow</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4260"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Generates specimen exception
		/// </summary>
		/// <param name="patientOrder"></param>
		/// <param name="specimen"></param>
		public void GenerateExceptionSpecimen(BOL.PatientOrder patientOrder, BOL.Specimen specimen) 
		{
			DateTime LastTransfusionDateTime = DateTime.MinValue;
			if (patientOrder != null)
			{
				LastTransfusionDateTime = BOL.PatientTransfusion.GetLastTransfusionDateTime(patientOrder.Patient.PatientGuid, VBECSDateTime.GetDivisionCurrentDateTime().AddMonths(-3), VBECSDateTime.GetDivisionCurrentDateTime());
			}
			BOL.OrderedTest OrderedTest = null;
			BOL.OrderedComponent OrderedComponent = null;
			string PatientAntibodies = "";
			string ComponentClass = "";

			DataTable dtTRs = BOL.PatientTransfusionRequirement.GetTransfusionRequirementsForPatient(specimen.Patient.PatientGuid);
			foreach(DataRow drTR in dtTRs.Rows)
			{
				if (drTR[TABLE.PatientTransfusionRequirement.TransfusionRequirementCategoryCode].ToString().Trim() == Common.Utility.GetTransfusionCategoryCodeFromEnum(Common.TransfusionRequirementCategoryCode.AntibodyIdentified))
				{
					PatientAntibodies += drTR[TABLE.PatientTransfusionRequirement.TransfusionRequirementText].ToString() + ", ";
				}
			}
			if (PatientAntibodies.Length > 0)
				PatientAntibodies = PatientAntibodies.Substring(0, PatientAntibodies.Length-2);

			if (patientOrder != null)
			{
				if (patientOrder is BOL.OrderedTest)
				{
					OrderedTest = (BOL.OrderedTest)patientOrder;
					ComponentClass = OrderedTest.TaskStatusText;
				}
				else
				{
					OrderedComponent = (BOL.OrderedComponent)patientOrder;
					ComponentClass = OrderedComponent.ComponentOrdered.BloodComponentName;
				}
			}

			DataTable dtExceptionSpecimen = new DataTable(TABLE.ExceptionSpecimen.TableName);
			dtExceptionSpecimen.Columns.Add(TABLE.ExceptionSpecimen.ExceptionSpecimenGuid, typeof(Guid));
			dtExceptionSpecimen.Columns.Add(TABLE.ExceptionSpecimen.ExceptionTypeId, typeof(int));
			dtExceptionSpecimen.Columns.Add(TABLE.ExceptionSpecimen.PatientGuid, typeof(Guid));
			dtExceptionSpecimen.Columns.Add(TABLE.ExceptionSpecimen.PatientName);
			dtExceptionSpecimen.Columns.Add(TABLE.ExceptionSpecimen.PatientId);
			dtExceptionSpecimen.Columns.Add(TABLE.ExceptionSpecimen.PatientAntibodies);
			dtExceptionSpecimen.Columns.Add(TABLE.ExceptionSpecimen.LastTransfusionDateTime);
			dtExceptionSpecimen.Columns.Add(TABLE.ExceptionSpecimen.SpecimenUid);
			dtExceptionSpecimen.Columns.Add(TABLE.ExceptionSpecimen.SpecimenExpirationDate, typeof(DateTime));
			dtExceptionSpecimen.Columns.Add(TABLE.ExceptionSpecimen.ExtendedExpirationDate, typeof(DateTime));
			dtExceptionSpecimen.Columns.Add(TABLE.ExceptionSpecimen.ComponentClassOrdered);
			dtExceptionSpecimen.Columns.Add(TABLE.ExceptionSpecimen.DateWanted, typeof(DateTime));
			dtExceptionSpecimen.Columns.Add(TABLE.ExceptionSpecimen.DateOrderAccepted, typeof(DateTime));
			dtExceptionSpecimen.Columns.Add(TABLE.ExceptionSpecimen.TestingTechName);
			dtExceptionSpecimen.Columns.Add(TABLE.ExceptionSpecimen.ExceptionUser);
			dtExceptionSpecimen.Columns.Add(TABLE.ExceptionSpecimen.ExceptionComment);
			dtExceptionSpecimen.Columns.Add(TABLE.ExceptionSpecimen.ExceptionDate, typeof(DateTime));
			dtExceptionSpecimen.Columns.Add(TABLE.ExceptionSpecimen.DivisionCode);

			this.ExceptionData = dtExceptionSpecimen.NewRow();
			this.ExceptionData[TABLE.ExceptionSpecimen.ExceptionSpecimenGuid] = Guid.NewGuid();
			this.ExceptionData[TABLE.ExceptionSpecimen.ExceptionTypeId] = this.ExceptionType;
			this.ExceptionData[TABLE.ExceptionSpecimen.PatientGuid] = specimen.Patient.PatientGuid;
			this.ExceptionData[TABLE.ExceptionSpecimen.PatientName] = specimen.Patient.DisplayName;
			this.ExceptionData[TABLE.ExceptionSpecimen.PatientId] = specimen.Patient.DisplayPatientID;
			this.ExceptionData[TABLE.ExceptionSpecimen.PatientAntibodies] = PatientAntibodies;
			this.ExceptionData[TABLE.ExceptionSpecimen.SpecimenUid] = specimen.SpecimenUid;
			if (specimen.PreviousExpirationDate == DateTime.MinValue)
			{
				this.ExceptionData[TABLE.ExceptionSpecimen.SpecimenExpirationDate] = specimen.SpecimenCollectionDate.Date.AddMinutes(-1).AddDays(Convert.ToDouble(Common.SpecimenExpirationDays.Minimum + 1));
			}
			else
			{
				this.ExceptionData[TABLE.ExceptionSpecimen.SpecimenExpirationDate] = specimen.PreviousExpirationDate;
			}
			if (this.ExceptionType == Common.ExceptionType.Specimen_expiration_date_extended)
			{
				this.ExceptionData[TABLE.ExceptionSpecimen.ExtendedExpirationDate] = specimen.SpecimenExpirationDate;
				this.ExceptionData[TABLE.ExceptionSpecimen.ComponentClassOrdered] = ComponentClass;
				if (patientOrder != null)
				{
					if (LastTransfusionDateTime == DateTime.MinValue)
					{
						this.ExceptionData[TABLE.ExceptionSpecimen.LastTransfusionDateTime] = DBNull.Value;
					}
					else
					{
						this.ExceptionData[TABLE.ExceptionSpecimen.LastTransfusionDateTime] = LastTransfusionDateTime;
					}

					if (OrderedTest != null)
					{
						if (OrderedTest.RequiredResultsDate != DateTime.MinValue)
							this.ExceptionData[TABLE.ExceptionSpecimen.DateWanted] = OrderedTest.RequiredResultsDate;
					}
					else if (OrderedComponent != null) //3554
					{
						if (OrderedComponent.RequiredDateTime != DateTime.MinValue)
							this.ExceptionData[TABLE.ExceptionSpecimen.DateWanted] = OrderedComponent.RequiredDateTime;
					}
					this.ExceptionData[TABLE.ExceptionSpecimen.DateOrderAccepted] = patientOrder.OrderReceivedDateTime;
				}
			}
			this.ExceptionData[TABLE.ExceptionSpecimen.TestingTechName] = this.ExceptionUser;
			this.ExceptionData[TABLE.ExceptionSpecimen.ExceptionUser] = this.ExceptionData[TABLE.ExceptionSpecimen.TestingTechName];
			this.ExceptionData[TABLE.ExceptionSpecimen.ExceptionComment] = this.ExceptionComment;
			this.ExceptionData[TABLE.ExceptionSpecimen.ExceptionDate] = this.ExceptionDate;
			this.ExceptionData[TABLE.ExceptionSpecimen.DivisionCode] = Common.LogonUser.LogonUserDivisionCode;
		}

		///<Developers>
		///	<Developer>Krzysztof Dobranowski</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/7/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4263"> 
		///		<ExpectedInput>Valid exception data</ExpectedInput>
		///		<ExpectedOutput>Null ExceptionData</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4264"> 
		///		<ExpectedInput>Null OrderedTest</ExpectedInput>
		///		<ExpectedOutput>Null ExceptionData</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Generates PatientAntigenDiscrepancy exception
		/// </summary>
		/// <param name="antigenOrder"></param>
		/// <param name="reagentTypeName"></param>
		/// <param name="reagentName"></param>
		/// <param name="lotNumber"></param>
		/// <param name="reagentManufacturerName"></param>
		/// <param name="lotExpirationDate"></param>
		/// <param name="historicalResult"></param>
		/// <param name="currentResult"></param>
		/// <param name="testingTech"></param>
		public void GenerateExceptionPatientAntigenDiscrepancy(BOL.OrderedTest antigenOrder, string reagentTypeName, string reagentName, string lotNumber, string reagentManufacturerName, DateTime? lotExpirationDate, 
			string historicalResult, string currentResult, string testingTech)
		{
			if (antigenOrder == null)
			{
				this.ExceptionData = null;
				return;
			}

			if (historicalResult.Trim() == "N")
				historicalResult = "Neg";
			else if (historicalResult.Trim() == "P")
				historicalResult = "Pos";
			else if (historicalResult.Trim() == "I")
				historicalResult = "Inc";

			if (currentResult.Trim() == "N")
				currentResult = "Neg";
			else if (currentResult.Trim() == "P")
				currentResult = "Pos";
			else if (currentResult.Trim() == "I")
				currentResult = "Inc";

			DataTable dtExceptionAntigenDiscrepancy = new DataTable(TABLE.ExceptionPatientAntigenDiscrepancy.TableName);
			dtExceptionAntigenDiscrepancy.Columns.Add(TABLE.ExceptionPatientAntigenDiscrepancy.ExceptionPatientAntigenDiscrepancyGuid, typeof(Guid));
			dtExceptionAntigenDiscrepancy.Columns.Add(TABLE.ExceptionPatientAntigenDiscrepancy.ExceptionTypeId, typeof(int));
			dtExceptionAntigenDiscrepancy.Columns.Add(TABLE.ExceptionPatientAntigenDiscrepancy.PatientGuid, typeof(Guid));
			dtExceptionAntigenDiscrepancy.Columns.Add(TABLE.ExceptionPatientAntigenDiscrepancy.PatientName);
			dtExceptionAntigenDiscrepancy.Columns.Add(TABLE.ExceptionPatientAntigenDiscrepancy.PatientId);
			dtExceptionAntigenDiscrepancy.Columns.Add(TABLE.ExceptionPatientAntigenDiscrepancy.SpecimenUid);
			dtExceptionAntigenDiscrepancy.Columns.Add(TABLE.ExceptionPatientAntigenDiscrepancy.ReagentName);
			dtExceptionAntigenDiscrepancy.Columns.Add(TABLE.ExceptionPatientAntigenDiscrepancy.ReagentTypeName);
			dtExceptionAntigenDiscrepancy.Columns.Add(TABLE.ExceptionPatientAntigenDiscrepancy.LotNumber);
			dtExceptionAntigenDiscrepancy.Columns.Add(TABLE.ExceptionPatientAntigenDiscrepancy.ReagentManufacturerName);
			dtExceptionAntigenDiscrepancy.Columns.Add(TABLE.ExceptionPatientAntigenDiscrepancy.LotExpirationDate, typeof(DateTime));
			dtExceptionAntigenDiscrepancy.Columns.Add(TABLE.ExceptionPatientAntigenDiscrepancy.HistoricSpecimenTestResults);
			dtExceptionAntigenDiscrepancy.Columns.Add(TABLE.ExceptionPatientAntigenDiscrepancy.CurrentSpecimenTestResults);
			dtExceptionAntigenDiscrepancy.Columns.Add(TABLE.ExceptionPatientAntigenDiscrepancy.TestingTechName);
			dtExceptionAntigenDiscrepancy.Columns.Add(TABLE.ExceptionPatientAntigenDiscrepancy.ExceptionUser);
			dtExceptionAntigenDiscrepancy.Columns.Add(TABLE.ExceptionPatientAntigenDiscrepancy.ExceptionComment);
			dtExceptionAntigenDiscrepancy.Columns.Add(TABLE.ExceptionPatientAntigenDiscrepancy.ExceptionDate, typeof(DateTime));
			dtExceptionAntigenDiscrepancy.Columns.Add(TABLE.ExceptionPatientAntigenDiscrepancy.DivisionCode);

			this.ExceptionData = dtExceptionAntigenDiscrepancy.NewRow();
			this.ExceptionData[TABLE.ExceptionPatientAntigenDiscrepancy.ExceptionPatientAntigenDiscrepancyGuid] = Guid.NewGuid();
			this.ExceptionData[TABLE.ExceptionPatientAntigenDiscrepancy.ExceptionTypeId] = this.ExceptionType;
			this.ExceptionData[TABLE.ExceptionPatientAntigenDiscrepancy.PatientGuid] = antigenOrder.Patient.PatientGuid;
			this.ExceptionData[TABLE.ExceptionPatientAntigenDiscrepancy.PatientName] = antigenOrder.Patient.DisplayName;
			this.ExceptionData[TABLE.ExceptionPatientAntigenDiscrepancy.PatientId] = antigenOrder.Patient.DisplayPatientID;
			this.ExceptionData[TABLE.ExceptionPatientAntigenDiscrepancy.SpecimenUid] = antigenOrder.Specimen.SpecimenUid;
			this.ExceptionData[TABLE.ExceptionPatientAntigenDiscrepancy.ReagentName] = reagentName;
			this.ExceptionData[TABLE.ExceptionPatientAntigenDiscrepancy.ReagentTypeName] = reagentTypeName;
			this.ExceptionData[TABLE.ExceptionPatientAntigenDiscrepancy.LotNumber] = lotNumber;
			this.ExceptionData[TABLE.ExceptionPatientAntigenDiscrepancy.ReagentManufacturerName] = reagentManufacturerName;
            this.ExceptionData[TABLE.ExceptionPatientAntigenDiscrepancy.LotExpirationDate] = DBNull.Value;
            if (lotExpirationDate.HasValue)
            {
                this.ExceptionData[TABLE.ExceptionPatientAntigenDiscrepancy.LotExpirationDate] = lotExpirationDate.Value;
            }
			this.ExceptionData[TABLE.ExceptionPatientAntigenDiscrepancy.HistoricSpecimenTestResults] = historicalResult;
			this.ExceptionData[TABLE.ExceptionPatientAntigenDiscrepancy.CurrentSpecimenTestResults] = currentResult;
			this.ExceptionData[TABLE.ExceptionPatientAntigenDiscrepancy.TestingTechName] = GetUserNameFromID(testingTech);
			this.ExceptionData[TABLE.ExceptionPatientAntigenDiscrepancy.ExceptionUser] = this.ExceptionUser;
			this.ExceptionData[TABLE.ExceptionPatientAntigenDiscrepancy.ExceptionComment] = this.ExceptionComment;
			this.ExceptionData[TABLE.ExceptionPatientAntigenDiscrepancy.ExceptionDate] = this.ExceptionDate;
			this.ExceptionData[TABLE.ExceptionPatientAntigenDiscrepancy.DivisionCode] = Common.LogonUser.LogonUserDivisionCode;
		}

		///<Developers>
		///	<Developer>Krzysztof Dobranowski</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/7/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4265"> 
		///		<ExpectedInput>Valid exception data</ExpectedInput>
		///		<ExpectedOutput>Valid exception DataRow</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4266"> 
		///		<ExpectedInput>Null PatientOrder</ExpectedInput>
		///		<ExpectedOutput>Null ExceptionData</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Generates exception for inactivated patient test
		/// </summary>
		/// <param name="patientOrder"></param>
		/// <param name="testingTechName"></param>
		public void GenerateExceptionPatientTestInactivated(BOL.PatientOrder patientOrder, string testingTechName)
		{
			BOL.OrderedTest orderedTest = null;
			BOL.OrderedComponent orderedComponent = null;

			if (patientOrder == null)
			{
				this.ExceptionData = null;
				return;
			}
			else
			{
				if (patientOrder is BOL.OrderedTest)
					orderedTest = (BOL.OrderedTest)patientOrder;
				else
					orderedComponent = (BOL.OrderedComponent)patientOrder;
			}

			DataTable dtExceptionPatientTestInactivated = new DataTable(TABLE.ExceptionPatientTestInactivated.TableName);
			dtExceptionPatientTestInactivated.Columns.Add(TABLE.ExceptionPatientTestInactivated.ExceptionPatientTestInactivatedGuid, typeof(Guid));
			dtExceptionPatientTestInactivated.Columns.Add(TABLE.ExceptionPatientTestInactivated.ExceptionTypeId, typeof(int));
			dtExceptionPatientTestInactivated.Columns.Add(TABLE.ExceptionPatientTestInactivated.PatientGuid, typeof(Guid));
			dtExceptionPatientTestInactivated.Columns.Add(TABLE.ExceptionPatientTestInactivated.PatientName);
			dtExceptionPatientTestInactivated.Columns.Add(TABLE.ExceptionPatientTestInactivated.PatientId);
			dtExceptionPatientTestInactivated.Columns.Add(TABLE.ExceptionPatientTestInactivated.SpecimenUid);
			dtExceptionPatientTestInactivated.Columns.Add(TABLE.ExceptionPatientTestInactivated.TestName);
			dtExceptionPatientTestInactivated.Columns.Add(TABLE.ExceptionPatientTestInactivated.InactivationDate, typeof(DateTime));
			dtExceptionPatientTestInactivated.Columns.Add(TABLE.ExceptionPatientTestInactivated.TestingTechName);
			dtExceptionPatientTestInactivated.Columns.Add(TABLE.ExceptionPatientTestInactivated.CurrentTechName);
			dtExceptionPatientTestInactivated.Columns.Add(TABLE.ExceptionPatientTestInactivated.ExceptionUser);
			dtExceptionPatientTestInactivated.Columns.Add(TABLE.ExceptionPatientTestInactivated.ExceptionComment);
			dtExceptionPatientTestInactivated.Columns.Add(TABLE.ExceptionPatientTestInactivated.ExceptionDate, typeof(DateTime));
			dtExceptionPatientTestInactivated.Columns.Add(TABLE.ExceptionPatientTestInactivated.DivisionCode);

			this.ExceptionData = dtExceptionPatientTestInactivated.NewRow();
			this.ExceptionData[TABLE.ExceptionPatientTestInactivated.ExceptionPatientTestInactivatedGuid] = Guid.NewGuid();
			this.ExceptionData[TABLE.ExceptionPatientTestInactivated.ExceptionTypeId] = this.ExceptionType;

			if (orderedTest != null)
			{
				this.ExceptionData[TABLE.ExceptionPatientTestInactivated.PatientGuid] = orderedTest.Patient.PatientGuid;
				this.ExceptionData[TABLE.ExceptionPatientTestInactivated.PatientName] = orderedTest.Patient.DisplayName;
				this.ExceptionData[TABLE.ExceptionPatientTestInactivated.PatientId] = orderedTest.Patient.DisplayPatientID;
				this.ExceptionData[TABLE.ExceptionPatientTestInactivated.SpecimenUid] = orderedTest.Specimen.SpecimenUid;
				this.ExceptionData[TABLE.ExceptionPatientTestInactivated.TestName] = orderedTest.OrderableTest;
			}
            else if (orderedComponent != null) //CR 3554
			{
				this.ExceptionData[TABLE.ExceptionPatientTestInactivated.PatientGuid] = orderedComponent.Patient.PatientGuid;
				this.ExceptionData[TABLE.ExceptionPatientTestInactivated.PatientName] = orderedComponent.Patient.DisplayName;
				this.ExceptionData[TABLE.ExceptionPatientTestInactivated.PatientId] = orderedComponent.Patient.DisplayPatientID;
				this.ExceptionData[TABLE.ExceptionPatientTestInactivated.SpecimenUid] = orderedComponent.GetSpecimen().SpecimenUid;  // CR3114
				this.ExceptionData[TABLE.ExceptionPatientTestInactivated.TestName] = "Crossmatch";
			}

			this.ExceptionData[TABLE.ExceptionPatientTestInactivated.InactivationDate] = this.ExceptionDate;
			this.ExceptionData[TABLE.ExceptionPatientTestInactivated.TestingTechName] = GetUserNameFromID(testingTechName);
			this.ExceptionData[TABLE.ExceptionPatientTestInactivated.CurrentTechName] = this.ExceptionUser;
			this.ExceptionData[TABLE.ExceptionPatientTestInactivated.ExceptionUser] = this.ExceptionUser;
			this.ExceptionData[TABLE.ExceptionPatientTestInactivated.ExceptionComment] = this.ExceptionComment;
			this.ExceptionData[TABLE.ExceptionPatientTestInactivated.ExceptionDate] = this.ExceptionDate;
			this.ExceptionData[TABLE.ExceptionPatientTestInactivated.DivisionCode] = Common.LogonUser.LogonUserDivisionCode;
		}

		///<Developers>
		///	<Developer>Krzysztof Dobranowski</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/7/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4267"> 
		///		<ExpectedInput>Valid exception data</ExpectedInput>
		///		<ExpectedOutput>Valid exception DataRow</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4268"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Generates exception for inconclusive Abo/Rh typing for patient
		/// </summary>
		/// <param name="patient"></param>
		/// <param name="specimenUID"></param>
		/// <param name="systemAbo"></param>
		/// <param name="systemRh"></param>
		/// <param name="techAbo"></param>
		/// <param name="techRh"></param>
		/// <param name="testingTech"></param>
		/// <param name="testDate"></param>
		public void GenerateExceptionPatientABORh(BOL.Patient patient, string specimenUID, string systemAbo, 
			string systemRh, string techAbo, string techRh, string testingTech, DateTime testDate)
		{
			string systemAboRh = CombineAboRh(systemAbo, systemRh);
			string techAboRh = CombineAboRh(techAbo, techRh);

			DataTable dtExceptionPatientAboRh = new DataTable(TABLE.ExceptionPatientAboRh.TableName);
			dtExceptionPatientAboRh.Columns.Add(TABLE.ExceptionPatientAboRh.ExceptionPatientAboRhGuid, typeof(Guid));
			dtExceptionPatientAboRh.Columns.Add(TABLE.ExceptionPatientAboRh.ExceptionTypeId, typeof(int));
			dtExceptionPatientAboRh.Columns.Add(TABLE.ExceptionPatientAboRh.PatientGuid, typeof(Guid));
			dtExceptionPatientAboRh.Columns.Add(TABLE.ExceptionPatientAboRh.PatientName);
			dtExceptionPatientAboRh.Columns.Add(TABLE.ExceptionPatientAboRh.PatientId);
			dtExceptionPatientAboRh.Columns.Add(TABLE.ExceptionPatientAboRh.SpecimenUid);
			dtExceptionPatientAboRh.Columns.Add(TABLE.ExceptionPatientAboRh.PreTestingAboRh);
			dtExceptionPatientAboRh.Columns.Add(TABLE.ExceptionPatientAboRh.CurrentSystemAboRh);
			dtExceptionPatientAboRh.Columns.Add(TABLE.ExceptionPatientAboRh.CurrentTechAboRh);
			dtExceptionPatientAboRh.Columns.Add(TABLE.ExceptionPatientAboRh.TestDate, typeof(DateTime));
			dtExceptionPatientAboRh.Columns.Add(TABLE.ExceptionPatientAboRh.TestingTechName);
			dtExceptionPatientAboRh.Columns.Add(TABLE.ExceptionPatientAboRh.ExceptionUser);
			dtExceptionPatientAboRh.Columns.Add(TABLE.ExceptionPatientAboRh.ExceptionComment);
			dtExceptionPatientAboRh.Columns.Add(TABLE.ExceptionPatientAboRh.ExceptionDate, typeof(DateTime));
			dtExceptionPatientAboRh.Columns.Add(TABLE.ExceptionPatientAboRh.DivisionCode);

			this.ExceptionData = dtExceptionPatientAboRh.NewRow();
			this.ExceptionData[TABLE.ExceptionPatientAboRh.ExceptionPatientAboRhGuid] = Guid.NewGuid();
			this.ExceptionData[TABLE.ExceptionPatientAboRh.ExceptionTypeId] = this.ExceptionType;
			this.ExceptionData[TABLE.ExceptionPatientAboRh.PatientGuid] = patient.PatientGuid;
			this.ExceptionData[TABLE.ExceptionPatientAboRh.PatientName] = patient.DisplayName;
			this.ExceptionData[TABLE.ExceptionPatientAboRh.PatientId] = patient.DisplayPatientID;
			this.ExceptionData[TABLE.ExceptionPatientAboRh.SpecimenUid] = specimenUID;
			this.ExceptionData[TABLE.ExceptionPatientAboRh.PreTestingAboRh] = patient.AboRh.AboRHText;
			this.ExceptionData[TABLE.ExceptionPatientAboRh.CurrentSystemAboRh] = systemAboRh;
			this.ExceptionData[TABLE.ExceptionPatientAboRh.CurrentTechAboRh] = techAboRh;
			this.ExceptionData[TABLE.ExceptionPatientAboRh.TestDate] = testDate;
			this.ExceptionData[TABLE.ExceptionPatientAboRh.TestingTechName] = GetUserNameFromID(testingTech);
			this.ExceptionData[TABLE.ExceptionPatientAboRh.ExceptionUser] = this.ExceptionUser;
			this.ExceptionData[TABLE.ExceptionPatientAboRh.ExceptionComment] = this.ExceptionComment;
			this.ExceptionData[TABLE.ExceptionPatientAboRh.ExceptionDate] = this.ExceptionDate;
			this.ExceptionData[TABLE.ExceptionPatientAboRh.DivisionCode] = Common.LogonUser.LogonUserDivisionCode;
		}

		///<Developers>
		///	<Developer>Krzysztof Dobranowski</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/7/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4269"> 
		///		<ExpectedInput>Valid exception data</ExpectedInput>
		///		<ExpectedOutput>Valid exception DataRow</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4270"> 
		///		<ExpectedInput>Null PatientOrder</ExpectedInput>
		///		<ExpectedOutput>Null ExceptionData</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Generates exception for non-standard test procedure
		/// </summary>
		/// <param name="patientOrder"></param>
		/// <param name="phaseOfReaction"></param>
		/// <param name="testInterp"></param>
		/// <param name="testDate"></param>
		/// <param name="testingTech"></param>
		public void GenerateExceptionPatientTest(BOL.PatientOrder patientOrder, string phaseOfReaction, string testInterp, DateTime testDate, string testingTech)
		{
			BOL.OrderedTest orderedTest = null;
			BOL.OrderedComponent orderedComponent = null;

			if (patientOrder == null)
			{
				this.ExceptionData = null;
				return;
			}
			else
			{
				if (patientOrder is BOL.OrderedTest)
				{
					orderedTest = (BOL.OrderedTest)patientOrder;
				}
				else
				{
					orderedComponent = (BOL.OrderedComponent)patientOrder;
				}
			}

			DataTable dtExceptionPatientTest = new DataTable(TABLE.ExceptionPatientTest.TableName);
			dtExceptionPatientTest.Columns.Add(TABLE.ExceptionPatientTest.ExceptionPatientTestGuid, typeof(Guid));
			dtExceptionPatientTest.Columns.Add(TABLE.ExceptionPatientTest.ExceptionTypeId, typeof(int));
			dtExceptionPatientTest.Columns.Add(TABLE.ExceptionPatientTest.PatientGuid, typeof(Guid));
			dtExceptionPatientTest.Columns.Add(TABLE.ExceptionPatientTest.PatientName);
			dtExceptionPatientTest.Columns.Add(TABLE.ExceptionPatientTest.PatientId);
			dtExceptionPatientTest.Columns.Add(TABLE.ExceptionPatientTest.SpecimenUid);
			dtExceptionPatientTest.Columns.Add(TABLE.ExceptionPatientTest.TestName);
			dtExceptionPatientTest.Columns.Add(TABLE.ExceptionPatientTest.TestInterp);
			dtExceptionPatientTest.Columns.Add(TABLE.ExceptionPatientTest.TestDate);
			dtExceptionPatientTest.Columns.Add(TABLE.ExceptionPatientTest.TestingTechName);
			dtExceptionPatientTest.Columns.Add(TABLE.ExceptionPatientTest.ReactivityPhase);
			dtExceptionPatientTest.Columns.Add(TABLE.ExceptionPatientTest.ExceptionUser);
			dtExceptionPatientTest.Columns.Add(TABLE.ExceptionPatientTest.ExceptionComment);
			dtExceptionPatientTest.Columns.Add(TABLE.ExceptionPatientTest.ExceptionDate, typeof(DateTime));
			dtExceptionPatientTest.Columns.Add(TABLE.ExceptionPatientTest.DivisionCode);

			this.ExceptionData = dtExceptionPatientTest.NewRow();
			this.ExceptionData[TABLE.ExceptionPatientTest.ExceptionPatientTestGuid] = Guid.NewGuid();
			this.ExceptionData[TABLE.ExceptionPatientTest.ExceptionTypeId] = this.ExceptionType;
			
			if (orderedTest != null)
			{
				this.ExceptionData[TABLE.ExceptionPatientTest.PatientGuid] = orderedTest.Patient.PatientGuid;
				this.ExceptionData[TABLE.ExceptionPatientTest.PatientName] = orderedTest.Patient.DisplayName;
				this.ExceptionData[TABLE.ExceptionPatientTest.PatientId] = orderedTest.Patient.DisplayPatientID;
				this.ExceptionData[TABLE.ExceptionPatientTest.SpecimenUid] = orderedTest.Specimen.SpecimenUid;
				this.ExceptionData[TABLE.ExceptionPatientTest.TestName] = orderedTest.OrderableTest;
			}
			else if(orderedComponent != null) //CR 3554
			{
				this.ExceptionData[TABLE.ExceptionPatientTest.PatientGuid] = orderedComponent.Patient.PatientGuid;
				this.ExceptionData[TABLE.ExceptionPatientTest.PatientName] = orderedComponent.Patient.DisplayName;
				this.ExceptionData[TABLE.ExceptionPatientTest.PatientId] = orderedComponent.Patient.DisplayPatientID;
				this.ExceptionData[TABLE.ExceptionPatientTest.SpecimenUid] = orderedComponent.GetSpecimen().SpecimenUid;  // CR3114
				this.ExceptionData[TABLE.ExceptionPatientTest.TestName] = "Crossmatch";
			}

			this.ExceptionData[TABLE.ExceptionPatientTest.ReactivityPhase] = phaseOfReaction;
			this.ExceptionData[TABLE.ExceptionPatientTest.TestInterp] = testInterp;
			this.ExceptionData[TABLE.ExceptionPatientTest.TestDate] = testDate;
			this.ExceptionData[TABLE.ExceptionPatientTest.TestingTechName] = GetUserNameFromID(testingTech);
			this.ExceptionData[TABLE.ExceptionPatientTest.ExceptionUser] = this.ExceptionUser;
			this.ExceptionData[TABLE.ExceptionPatientTest.ExceptionComment] = this.ExceptionComment;
			this.ExceptionData[TABLE.ExceptionPatientTest.ExceptionDate] = this.ExceptionDate;
			this.ExceptionData[TABLE.ExceptionPatientTest.DivisionCode] = Common.LogonUser.LogonUserDivisionCode;
		}

		///<Developers>
		///	<Developer>Krzysztof Dobranowski</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/7/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4271"> 
		///		<ExpectedInput>Valid exception data</ExpectedInput>
		///		<ExpectedOutput>Valid exception DataRow</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4272"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Generates exception when unit is quarantined, discarded or removed from quarantine
		/// </summary>
		/// <param name="bloodUnit"></param>
		public void GenerateExceptionQuarantine(BOL.BloodUnit bloodUnit)
		{
			BOL.Patient patient = null;
			if (bloodUnit.BloodUnitMedia.DirectedPatientGuid != Guid.Empty)
			{
				patient = new BOL.Patient(bloodUnit.BloodUnitMedia.DirectedPatientGuid);
			}

			string unitAntigens = "";
			ArrayList antigens = BOL.BloodUnitAntigen.GetBloodUnitAntigensList(bloodUnit.BloodUnitGuid, Common.LogonUser.LogonUserDivisionCode);
			foreach(BOL.BloodUnitAntigen antigen in antigens)
			{
				if (unitAntigens != "")
					unitAntigens += ", ";

				if (antigen.Positive)
				{
					unitAntigens += antigen.AntigenName + " Pos";
				}
				else
				{
					unitAntigens += antigen.AntigenName + " Neg";
				}
			}

			DataTable dtExceptionQuarantine = new DataTable(TABLE.ExceptionQuarantine.TableName);
			dtExceptionQuarantine.Columns.Add(TABLE.ExceptionQuarantine.ExceptionQuarantineGuid, typeof(Guid));
			dtExceptionQuarantine.Columns.Add(TABLE.ExceptionQuarantine.ExceptionTypeId, typeof(int));
			dtExceptionQuarantine.Columns.Add(TABLE.ExceptionQuarantine.PatientGuid, typeof(Guid));
			dtExceptionQuarantine.Columns.Add(TABLE.ExceptionQuarantine.PatientName);
			dtExceptionQuarantine.Columns.Add(TABLE.ExceptionQuarantine.PatientId);
			dtExceptionQuarantine.Columns.Add(TABLE.ExceptionQuarantine.UnitId);
			dtExceptionQuarantine.Columns.Add(TABLE.ExceptionQuarantine.UnitGuid);
			dtExceptionQuarantine.Columns.Add(TABLE.ExceptionQuarantine.UnitProductType);
			dtExceptionQuarantine.Columns.Add(TABLE.ExceptionQuarantine.UnitProductCode);
			dtExceptionQuarantine.Columns.Add(TABLE.ExceptionQuarantine.UnitAboRh);
			dtExceptionQuarantine.Columns.Add(TABLE.ExceptionQuarantine.UnitExpirationDate);
			dtExceptionQuarantine.Columns.Add(TABLE.ExceptionQuarantine.UnitAntigenTypingResult);
			dtExceptionQuarantine.Columns.Add(TABLE.ExceptionQuarantine.TestingTechName);
			dtExceptionQuarantine.Columns.Add(TABLE.ExceptionQuarantine.ExceptionUser);
			dtExceptionQuarantine.Columns.Add(TABLE.ExceptionQuarantine.ExceptionComment);
			dtExceptionQuarantine.Columns.Add(TABLE.ExceptionQuarantine.ExceptionDate, typeof(DateTime));
			dtExceptionQuarantine.Columns.Add(TABLE.ExceptionQuarantine.DivisionCode);

			this.ExceptionData = dtExceptionQuarantine.NewRow();
			this.ExceptionData[TABLE.ExceptionQuarantine.ExceptionQuarantineGuid] = Guid.NewGuid();
			this.ExceptionData[TABLE.ExceptionQuarantine.ExceptionTypeId] = this.ExceptionType;
			if (patient != null)
			{
				this.ExceptionData[TABLE.ExceptionQuarantine.PatientGuid] = patient.PatientGuid;
				this.ExceptionData[TABLE.ExceptionQuarantine.PatientName] = patient.DisplayName;
				this.ExceptionData[TABLE.ExceptionQuarantine.PatientId] = patient.DisplayPatientID;
			}
			this.ExceptionData[TABLE.ExceptionQuarantine.UnitId] = bloodUnit.EyeReadableUnitId;
			this.ExceptionData[TABLE.ExceptionQuarantine.UnitGuid] = bloodUnit.BloodUnitGuid;
			this.ExceptionData[TABLE.ExceptionQuarantine.UnitProductType] = bloodUnit.ProductType.ProductTypeText;
			this.ExceptionData[TABLE.ExceptionQuarantine.UnitProductCode] = bloodUnit.FullProductCode;
			this.ExceptionData[TABLE.ExceptionQuarantine.UnitAboRh] = bloodUnit.BloodUnitMedia.AboRh.AboRHText;
			this.ExceptionData[TABLE.ExceptionQuarantine.UnitExpirationDate] = bloodUnit.BloodUnitMedia.UnitExpirationDate;
			this.ExceptionData[TABLE.ExceptionQuarantine.UnitAntigenTypingResult] = unitAntigens;
			this.ExceptionData[TABLE.ExceptionQuarantine.TestingTechName] = this.ExceptionUser;
			this.ExceptionData[TABLE.ExceptionQuarantine.ExceptionUser] = this.ExceptionData[TABLE.ExceptionQuarantine.TestingTechName];
			this.ExceptionData[TABLE.ExceptionQuarantine.ExceptionComment] = this.ExceptionComment;
			this.ExceptionData[TABLE.ExceptionQuarantine.ExceptionDate] = this.ExceptionDate;
			this.ExceptionData[TABLE.ExceptionQuarantine.DivisionCode] = Common.LogonUser.LogonUserDivisionCode;
		}

		///<Developers>
		///	<Developer>Krzysztof Dobranowski</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/7/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4273"> 
		///		<ExpectedInput>Valid exception data</ExpectedInput>
		///		<ExpectedOutput>Valid exception DataRow</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4274"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Generate exception for Abo/Rh confirmation
		/// </summary>
		/// <param name="bloodUnitGuid"></param>
		/// <param name="aboInterp"></param>
		/// <param name="rhInterp"></param>
		/// <param name="testingTech"></param>
		public void GenerateExceptionUnitAboRh(Guid bloodUnitGuid, string aboInterp, string rhInterp, string testingTech)
		{
			BOL.BloodUnit bloodUnit = new BOL.BloodUnit(bloodUnitGuid);
			string aboRhInterp = CombineAboRh(aboInterp, rhInterp);

			DataTable dtExceptionUnitAboRh = new DataTable(TABLE.ExceptionUnitAboRh.TableName);
			dtExceptionUnitAboRh.Columns.Add(TABLE.ExceptionUnitAboRh.ExceptionUnitAboRhGuid, typeof(Guid));
			dtExceptionUnitAboRh.Columns.Add(TABLE.ExceptionUnitAboRh.ExceptionTypeId, typeof(int));
			dtExceptionUnitAboRh.Columns.Add(TABLE.ExceptionUnitAboRh.UnitId);
			dtExceptionUnitAboRh.Columns.Add(TABLE.ExceptionUnitAboRh.UnitGuid);
			dtExceptionUnitAboRh.Columns.Add(TABLE.ExceptionUnitAboRh.UnitProductType);
			dtExceptionUnitAboRh.Columns.Add(TABLE.ExceptionUnitAboRh.UnitExpirationDate, typeof(DateTime));
			dtExceptionUnitAboRh.Columns.Add(TABLE.ExceptionUnitAboRh.LoginUnitAboRh);
			dtExceptionUnitAboRh.Columns.Add(TABLE.ExceptionUnitAboRh.UnitAboRhInterp);
			dtExceptionUnitAboRh.Columns.Add(TABLE.ExceptionUnitAboRh.TestingTechName);
			dtExceptionUnitAboRh.Columns.Add(TABLE.ExceptionUnitAboRh.ExceptionUser);
			dtExceptionUnitAboRh.Columns.Add(TABLE.ExceptionUnitAboRh.ExceptionComment);
			dtExceptionUnitAboRh.Columns.Add(TABLE.ExceptionUnitAboRh.ExceptionDate, typeof(DateTime));
			dtExceptionUnitAboRh.Columns.Add(TABLE.ExceptionUnitAboRh.DivisionCode);

			this.ExceptionData = dtExceptionUnitAboRh.NewRow();
			this.ExceptionData[TABLE.ExceptionUnitAboRh.ExceptionUnitAboRhGuid] = Guid.NewGuid();
			this.ExceptionData[TABLE.ExceptionUnitAboRh.ExceptionTypeId] = this.ExceptionType;
			this.ExceptionData[TABLE.ExceptionUnitAboRh.UnitId] = bloodUnit.EyeReadableUnitId;
			this.ExceptionData[TABLE.ExceptionUnitAboRh.UnitGuid] = bloodUnit.BloodUnitGuid;
			this.ExceptionData[TABLE.ExceptionUnitAboRh.UnitProductType] = bloodUnit.ProductType.ProductTypeText;
			this.ExceptionData[TABLE.ExceptionUnitAboRh.UnitExpirationDate] = bloodUnit.BloodUnitMedia.UnitExpirationDate;
			this.ExceptionData[TABLE.ExceptionUnitAboRh.LoginUnitAboRh] = bloodUnit.BloodUnitMedia.AboRh.AboRHText;
			this.ExceptionData[TABLE.ExceptionUnitAboRh.UnitAboRhInterp] = aboRhInterp;
			this.ExceptionData[TABLE.ExceptionUnitAboRh.TestingTechName] = GetUserNameFromID(testingTech);
			this.ExceptionData[TABLE.ExceptionUnitAboRh.ExceptionUser] = this.ExceptionUser;
			this.ExceptionData[TABLE.ExceptionUnitAboRh.ExceptionComment] = this.ExceptionComment;
			this.ExceptionData[TABLE.ExceptionUnitAboRh.ExceptionDate] = this.ExceptionDate;
			this.ExceptionData[TABLE.ExceptionUnitAboRh.DivisionCode] = Common.LogonUser.LogonUserDivisionCode;
		}

		///<Developers>
		///	<Developer>Krzysztof Dobranowski</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/14/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4351"> 
		///		<ExpectedInput>Valid exception data</ExpectedInput>
		///		<ExpectedOutput>Valid exception DataRow</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4352"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Generates exception for blood unit modification
		/// </summary>
		/// <param name="bloodUnit"></param>
		/// <param name="bumControl"></param>
		/// <param name="recommendedExpirationDate"></param>
		/// <param name="supply"></param> 
		public void GenerateExceptionUnitModification(BOL.BloodUnit bloodUnit, BOL.BloodUnitModification bumControl, DateTime recommendedExpirationDate, BOL.Supply supply)
		{
			DataTable dtExceptionUnitModification = new DataTable(TABLE.ExceptionUnitModification.TableName);
			dtExceptionUnitModification.Columns.Add(TABLE.ExceptionUnitModification.ExceptionUnitModificationGuid, typeof(Guid));
			dtExceptionUnitModification.Columns.Add(TABLE.ExceptionUnitModification.ExceptionTypeId, typeof(int));
			dtExceptionUnitModification.Columns.Add(TABLE.ExceptionUnitModification.UnitId);
			dtExceptionUnitModification.Columns.Add(TABLE.ExceptionUnitModification.UnitGuid);
			dtExceptionUnitModification.Columns.Add(TABLE.ExceptionUnitModification.UnitAboRh);
			dtExceptionUnitModification.Columns.Add(TABLE.ExceptionUnitModification.UnitProductType);
			dtExceptionUnitModification.Columns.Add(TABLE.ExceptionUnitModification.UnitProductCode);
			dtExceptionUnitModification.Columns.Add(TABLE.ExceptionUnitModification.UnitExpirationDate, typeof(DateTime));
			dtExceptionUnitModification.Columns.Add(TABLE.ExceptionUnitModification.ModificationType);
			dtExceptionUnitModification.Columns.Add(TABLE.ExceptionUnitModification.ModificationDate, typeof(DateTime));
			dtExceptionUnitModification.Columns.Add(TABLE.ExceptionUnitModification.ModifyingTech);
			dtExceptionUnitModification.Columns.Add(TABLE.ExceptionUnitModification.ComponentName);
			dtExceptionUnitModification.Columns.Add(TABLE.ExceptionUnitModification.RecommendedExpirationDate, typeof(DateTime));
			dtExceptionUnitModification.Columns.Add(TABLE.ExceptionUnitModification.ExceptionUser);
			dtExceptionUnitModification.Columns.Add(TABLE.ExceptionUnitModification.ExceptionComment);
			dtExceptionUnitModification.Columns.Add(TABLE.ExceptionUnitModification.ExceptionDate, typeof(DateTime));
			dtExceptionUnitModification.Columns.Add(TABLE.ExceptionUnitModification.DivisionCode);

			//4 new fields for Supply portion
			dtExceptionUnitModification.Columns.Add(TABLE.ExceptionUnitModification.SupplyExpirationDate, typeof(DateTime));
			dtExceptionUnitModification.Columns.Add(TABLE.ExceptionUnitModification.SupplyLotNumber);
			dtExceptionUnitModification.Columns.Add(TABLE.ExceptionUnitModification.SupplyName);
			dtExceptionUnitModification.Columns.Add(TABLE.ExceptionUnitModification.SupplyType);

			this.ExceptionData = dtExceptionUnitModification.NewRow();
			this.ExceptionData[TABLE.ExceptionUnitModification.ExceptionUnitModificationGuid] = Guid.NewGuid();
			this.ExceptionData[TABLE.ExceptionUnitModification.ExceptionTypeId] = this.ExceptionType;
			this.ExceptionData[TABLE.ExceptionUnitModification.UnitId] = bloodUnit.EyeReadableUnitId;
			this.ExceptionData[TABLE.ExceptionUnitModification.UnitGuid] = bloodUnit.BloodUnitGuid;
			this.ExceptionData[TABLE.ExceptionUnitModification.UnitAboRh] = bloodUnit.BloodUnitMedia.AboRh.AboRHText;
			this.ExceptionData[TABLE.ExceptionUnitModification.UnitProductType] = bloodUnit.ProductType.ProductTypeText;
			this.ExceptionData[TABLE.ExceptionUnitModification.UnitProductCode] = bloodUnit.FullProductCode;
			this.ExceptionData[TABLE.ExceptionUnitModification.UnitExpirationDate] = bloodUnit.BloodUnitMedia.UnitExpirationDate;
			this.ExceptionData[TABLE.ExceptionUnitModification.ModificationType] = bumControl.ProductModificationCodeBase.ToString();
			this.ExceptionData[TABLE.ExceptionUnitModification.ModificationDate] = bumControl.ModificationDateTime;
			this.ExceptionData[TABLE.ExceptionUnitModification.ModifyingTech] = GetUserNameFromID(bumControl.ModificationTechId);
			this.ExceptionData[TABLE.ExceptionUnitModification.ComponentName] = bloodUnit.BloodComponentName;
			if (recommendedExpirationDate != DateTime.MinValue)
			{
				this.ExceptionData[TABLE.ExceptionUnitModification.RecommendedExpirationDate] = recommendedExpirationDate;
			}
			this.ExceptionData[TABLE.ExceptionUnitModification.ExceptionUser] = this.ExceptionUser;
			this.ExceptionData[TABLE.ExceptionUnitModification.ExceptionComment] = this.ExceptionComment;
			this.ExceptionData[TABLE.ExceptionUnitModification.ExceptionDate] = this.ExceptionDate;
			this.ExceptionData[TABLE.ExceptionUnitModification.DivisionCode] = Common.LogonUser.LogonUserDivisionCode;

			//If supply = null, then it's a regular MODIFY exception and supply details aren't important
			if (supply != null)
			{
				this.ExceptionData[TABLE.ExceptionUnitModification.SupplyExpirationDate] = supply.LotExpirationDate;
				this.ExceptionData[TABLE.ExceptionUnitModification.SupplyLotNumber] = supply.LotNumber;
				this.ExceptionData[TABLE.ExceptionUnitModification.SupplyName] = supply.SupplyName;
				this.ExceptionData[TABLE.ExceptionUnitModification.SupplyType] = supply.SupplyTypeText;
			}

		}

		///<Developers>
		///	<Developer>Krzysztof Dobranowski</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/17/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4432"> 
		///		<ExpectedInput>Valid exception data</ExpectedInput>
		///		<ExpectedOutput>Valid exception DataRow</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4433"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Generates exception for blood unit selection
		/// </summary>
		/// <param name="bloodUnitGuid"></param>
		/// <param name="patient"></param>
		/// <param name="selectionDate"></param>
		/// <param name="testingTech"></param>
		public void GenerateExceptionUnitSelection(Guid bloodUnitGuid, BOL.Patient patient, DateTime selectionDate, string testingTech)
		{
			BOL.BloodUnit bloodUnit = new BOL.BloodUnit(bloodUnitGuid);

			string patientAntibodies = "";
			string componentTRs = "";
			string patientAntigenNeg = "";
			string unitAntigens = "";

			//CR 2910 Added conversion antibodies
			DataRow[] drPatientTRs = patient.TransfusionRequirements.Select(TABLE.PatientTransfusionRequirement.RecordStatusCode + " = 'A' OR " + TABLE.PatientTransfusionRequirement.RecordStatusCode + " = 'C'", TABLE.PatientTransfusionRequirement.TransfusionRequirementCategoryCode);

			foreach(DataRow drPatientTR in drPatientTRs)
			{
				//antibodies identified
				if (drPatientTR[TABLE.PatientTransfusionRequirement.TransfusionRequirementCategoryCode].ToString().Trim() == Common.Utility.GetTransfusionCategoryCodeFromEnum(Common.TransfusionRequirementCategoryCode.AntibodyIdentified))
				{
					if (patientAntibodies != "")
					{
						patientAntibodies += ", ";
					}
					patientAntibodies += drPatientTR[TABLE.PatientTransfusionRequirement.TransfusionRequirementText].ToString().Trim();
				}
				if (drPatientTR[TABLE.PatientTransfusionRequirement.TransfusionRequirementCategoryCode].ToString().Trim() == Common.Utility.GetTransfusionCategoryCodeFromEnum(Common.TransfusionRequirementCategoryCode.AntigenNegative))
				{
					if (patientAntigenNeg != "")
					{
						patientAntigenNeg += ", ";
					}
					patientAntigenNeg += drPatientTR[TABLE.PatientTransfusionRequirement.TransfusionRequirementText].ToString().Trim();
				}
					//antigen negative and component TRs
				else if (drPatientTR[TABLE.PatientTransfusionRequirement.TransfusionRequirementCategoryCode].ToString().Trim() == Common.Utility.GetTransfusionCategoryCodeFromEnum(Common.TransfusionRequirementCategoryCode.ComponentRequirement))
				{
					if (componentTRs != "")
					{
						componentTRs += ", ";
					}
					componentTRs += drPatientTR[TABLE.PatientTransfusionRequirement.TransfusionRequirementText].ToString().Trim();
				}
			}
			
			DataTable dtBloodUnitAntigens = BOL.BloodUnitAntigen.GetBloodUnitAntigens(bloodUnitGuid, Common.LogonUser.LogonUserDivisionCode);
			foreach(DataRow drBloodUnitAntigen in dtBloodUnitAntigens.Rows)
			{
				if (unitAntigens != "")
				{
					unitAntigens += ", ";
				}

				unitAntigens += drBloodUnitAntigen[TABLE.AntigenType.AntigenTypeName].ToString();
				if ((bool)drBloodUnitAntigen[TABLE.BloodUnitAntigen.Positive] == true)
				{
					unitAntigens += " Pos";
				}
				else
				{
					unitAntigens += " Neg";
				}
			}

			ArrayList bloodUnitAntigenTypingTests = BOL.BloodUnitTest.GetAntigenTypingTests(bloodUnitGuid, Common.LogonUser.LogonUserDivisionCode);
			foreach(BOL.BloodUnitTest bloodUnitTest in bloodUnitAntigenTypingTests)
			{
				if (bloodUnitTest.BloodTestTypeObj.BloodTestName.IndexOf(ARTIFICIAL.Interp) != -1)
				{
					if (unitAntigens != "")
					{
						unitAntigens += ", ";
					}

					unitAntigens += bloodUnitTest.Specificity;
					if (bloodUnitTest.TestResultId.Trim() == "P")
					{
						unitAntigens += " Pos";
					}
					else if (bloodUnitTest.TestResultId.Trim() == "N")
					{
						unitAntigens += " Neg";
					}
					else if (bloodUnitTest.TestResultId.Trim() == "I")
					{
						unitAntigens += " Inc";
					}
				}
			}

			DataTable dtExceptionUnitSelection = new DataTable(TABLE.ExceptionUnitSelection.TableName);
			dtExceptionUnitSelection.Columns.Add(TABLE.ExceptionUnitSelection.ExceptionUnitSelectionGuid, typeof(Guid));
			dtExceptionUnitSelection.Columns.Add(TABLE.ExceptionUnitSelection.ExceptionTypeId, typeof(int));
			dtExceptionUnitSelection.Columns.Add(TABLE.ExceptionUnitSelection.PatientGuid, typeof(Guid));
			dtExceptionUnitSelection.Columns.Add(TABLE.ExceptionUnitSelection.PatientName);
			dtExceptionUnitSelection.Columns.Add(TABLE.ExceptionUnitSelection.PatientId);
			dtExceptionUnitSelection.Columns.Add(TABLE.ExceptionUnitSelection.PatientCurrentAboRh);
			dtExceptionUnitSelection.Columns.Add(TABLE.ExceptionUnitSelection.PatientHistoricalAboRh);
			dtExceptionUnitSelection.Columns.Add(TABLE.ExceptionUnitSelection.PatientAntibodies);
			dtExceptionUnitSelection.Columns.Add(TABLE.ExceptionUnitSelection.ComponentRequirements);
			dtExceptionUnitSelection.Columns.Add(TABLE.ExceptionUnitSelection.AntigenNegativeRequirements);
			dtExceptionUnitSelection.Columns.Add(TABLE.ExceptionUnitSelection.UnitId);
			dtExceptionUnitSelection.Columns.Add(TABLE.ExceptionUnitSelection.UnitGuid,typeof(Guid));
			dtExceptionUnitSelection.Columns.Add(TABLE.ExceptionUnitSelection.UnitProductType);
			dtExceptionUnitSelection.Columns.Add(TABLE.ExceptionUnitSelection.UnitAboRh);
			dtExceptionUnitSelection.Columns.Add(TABLE.ExceptionUnitSelection.UnitExpirationDate, typeof(DateTime));
			dtExceptionUnitSelection.Columns.Add(TABLE.ExceptionUnitSelection.AntigenTypingResult);
			dtExceptionUnitSelection.Columns.Add(TABLE.ExceptionUnitSelection.DonationType);
			dtExceptionUnitSelection.Columns.Add(TABLE.ExceptionUnitSelection.SelectionDate, typeof(DateTime));
			dtExceptionUnitSelection.Columns.Add(TABLE.ExceptionUnitSelection.TestingTechName);
			dtExceptionUnitSelection.Columns.Add(TABLE.ExceptionUnitSelection.ExceptionUser);
			dtExceptionUnitSelection.Columns.Add(TABLE.ExceptionUnitSelection.ExceptionComment);
			dtExceptionUnitSelection.Columns.Add(TABLE.ExceptionUnitSelection.ExceptionDate, typeof(DateTime));
			dtExceptionUnitSelection.Columns.Add(TABLE.ExceptionUnitSelection.DivisionCode);

			this.ExceptionData = dtExceptionUnitSelection.NewRow();
			this.ExceptionData[TABLE.ExceptionUnitSelection.ExceptionUnitSelectionGuid] = Guid.NewGuid();
			this.ExceptionData[TABLE.ExceptionUnitSelection.ExceptionTypeId] = this.ExceptionType;
			this.ExceptionData[TABLE.ExceptionUnitSelection.PatientGuid] = patient.PatientGuid;
			this.ExceptionData[TABLE.ExceptionUnitSelection.PatientName] = patient.DisplayName;
			this.ExceptionData[TABLE.ExceptionUnitSelection.PatientId] = patient.DisplayPatientID;
			this.ExceptionData[TABLE.ExceptionUnitSelection.PatientCurrentAboRh] = patient.AboRh.AboRHText;
			this.ExceptionData[TABLE.ExceptionUnitSelection.PatientHistoricalAboRh] = patient.PreviousAboRH.AboRHText;
			this.ExceptionData[TABLE.ExceptionUnitSelection.PatientAntibodies] = patientAntibodies;
			this.ExceptionData[TABLE.ExceptionUnitSelection.AntigenNegativeRequirements] = patientAntigenNeg;
			this.ExceptionData[TABLE.ExceptionUnitSelection.ComponentRequirements] = componentTRs;
			this.ExceptionData[TABLE.ExceptionUnitSelection.UnitId] = bloodUnit.EyeReadableUnitId;
			this.ExceptionData[TABLE.ExceptionUnitSelection.UnitGuid] = bloodUnit.BloodUnitGuid;
			this.ExceptionData[TABLE.ExceptionUnitSelection.UnitProductType] = bloodUnit.ProductType.ProductTypeText;
			this.ExceptionData[TABLE.ExceptionUnitSelection.UnitAboRh] = bloodUnit.BloodUnitMedia.AboRh.AboRHText;
			this.ExceptionData[TABLE.ExceptionUnitSelection.UnitExpirationDate] = bloodUnit.BloodUnitMedia.UnitExpirationDate;
			this.ExceptionData[TABLE.ExceptionUnitSelection.AntigenTypingResult] = unitAntigens;
			this.ExceptionData[TABLE.ExceptionUnitSelection.DonationType] = bloodUnit.DonationType.DonationTypeText;
			this.ExceptionData[TABLE.ExceptionUnitSelection.SelectionDate] = selectionDate;
			this.ExceptionData[TABLE.ExceptionUnitSelection.ExceptionUser] = this.ExceptionUser;
			if (testingTech != "")
			{
				this.ExceptionData[TABLE.ExceptionUnitSelection.TestingTechName] = GetUserNameFromID(testingTech);
			}
			else
			{
				this.ExceptionData[TABLE.ExceptionUnitSelection.TestingTechName] = this.ExceptionData[TABLE.ExceptionUnitSelection.ExceptionUser];
			}
			this.ExceptionData[TABLE.ExceptionUnitSelection.ExceptionComment] = this.ExceptionComment;
			this.ExceptionData[TABLE.ExceptionUnitSelection.ExceptionDate] = this.ExceptionDate;
			this.ExceptionData[TABLE.ExceptionUnitSelection.DivisionCode] = Common.LogonUser.LogonUserDivisionCode;
		}

		///<Developers>
		///	<Developer>Krzysztof Dobranowski</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>6/1/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4630"> 
		///		<ExpectedInput>Valid exception data</ExpectedInput>
		///		<ExpectedOutput>Valid exception DataRow</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4631"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Generates exception when there is a mismatch in label verification
		/// </summary>
		/// <param name="bloodUnit"></param>
		/// <param name="modificationType"></param>
		/// <param name="modifyingTech"></param>
		/// <param name="modificationDate"></param>
		/// <param name="expectedValue"></param>
		/// <param name="discrepantValue"></param>
		public void GenerateExceptionLabelMismatch(BOL.BloodUnit bloodUnit, string modificationType, string modifyingTech, DateTime modificationDate, string expectedValue, string discrepantValue)
		{
			DataTable dtExceptionLabelMismatch = new DataTable(TABLE.ExceptionLabelMismatch.TableName);
			dtExceptionLabelMismatch.Columns.Add(TABLE.ExceptionLabelMismatch.ExceptionLabelMismatchGuid, typeof(Guid));
			dtExceptionLabelMismatch.Columns.Add(TABLE.ExceptionLabelMismatch.ExceptionTypeId, typeof(int));
			dtExceptionLabelMismatch.Columns.Add(TABLE.ExceptionLabelMismatch.TargetUnitId);
			dtExceptionLabelMismatch.Columns.Add(TABLE.ExceptionLabelMismatch.TargetUnitGuid);
			dtExceptionLabelMismatch.Columns.Add(TABLE.ExceptionLabelMismatch.TargetUnitAboRh);
			dtExceptionLabelMismatch.Columns.Add(TABLE.ExceptionLabelMismatch.TargetProductType);
			dtExceptionLabelMismatch.Columns.Add(TABLE.ExceptionLabelMismatch.TargetProductCode);
			dtExceptionLabelMismatch.Columns.Add(TABLE.ExceptionLabelMismatch.TargetUnitExpirationDate, typeof(DateTime));
			dtExceptionLabelMismatch.Columns.Add(TABLE.ExceptionLabelMismatch.ModificationType);
			dtExceptionLabelMismatch.Columns.Add(TABLE.ExceptionLabelMismatch.ModificationDate, typeof(DateTime));
			dtExceptionLabelMismatch.Columns.Add(TABLE.ExceptionLabelMismatch.DiscrepantField);
			dtExceptionLabelMismatch.Columns.Add(TABLE.ExceptionLabelMismatch.CorrectedValue);
			dtExceptionLabelMismatch.Columns.Add(TABLE.ExceptionLabelMismatch.TestingTechName);
			dtExceptionLabelMismatch.Columns.Add(TABLE.ExceptionLabelMismatch.ExceptionUser);
			dtExceptionLabelMismatch.Columns.Add(TABLE.ExceptionLabelMismatch.ExceptionComment);
			dtExceptionLabelMismatch.Columns.Add(TABLE.ExceptionLabelMismatch.ExceptionDate, typeof(DateTime));
			dtExceptionLabelMismatch.Columns.Add(TABLE.ExceptionLabelMismatch.DivisionCode);

			this.ExceptionData = dtExceptionLabelMismatch.NewRow();
			this.ExceptionData[TABLE.ExceptionLabelMismatch.ExceptionLabelMismatchGuid] = Guid.NewGuid();
			this.ExceptionData[TABLE.ExceptionLabelMismatch.ExceptionTypeId] = this.ExceptionType;
			this.ExceptionData[TABLE.ExceptionLabelMismatch.TargetUnitId] = bloodUnit.EyeReadableUnitId;
			this.ExceptionData[TABLE.ExceptionLabelMismatch.TargetUnitGuid] = bloodUnit.BloodUnitGuid;
			this.ExceptionData[TABLE.ExceptionLabelMismatch.TargetUnitAboRh] = bloodUnit.BloodUnitMedia.AboRh.AboRHText;
			this.ExceptionData[TABLE.ExceptionLabelMismatch.TargetProductType] = bloodUnit.ProductType.ProductTypeText;
			this.ExceptionData[TABLE.ExceptionLabelMismatch.TargetProductCode] = bloodUnit.FullProductCode;
			this.ExceptionData[TABLE.ExceptionLabelMismatch.TargetUnitExpirationDate] = bloodUnit.BloodUnitMedia.UnitExpirationDate;
			this.ExceptionData[TABLE.ExceptionLabelMismatch.ModificationType] = modificationType;
			this.ExceptionData[TABLE.ExceptionLabelMismatch.ModificationDate] = modificationDate;
			if (discrepantValue.Trim() != "")
			{
				this.ExceptionData[TABLE.ExceptionLabelMismatch.DiscrepantField] = discrepantValue;
			}
			else
			{
				this.ExceptionData[TABLE.ExceptionLabelMismatch.DiscrepantField] = "no entry";
			}
			this.ExceptionData[TABLE.ExceptionLabelMismatch.CorrectedValue] = expectedValue;
			this.ExceptionData[TABLE.ExceptionLabelMismatch.TestingTechName] = GetUserNameFromID(modifyingTech);
			this.ExceptionData[TABLE.ExceptionLabelMismatch.ExceptionUser] = this.ExceptionUser;
			this.ExceptionData[TABLE.ExceptionLabelMismatch.ExceptionComment] = this.ExceptionComment;
			this.ExceptionData[TABLE.ExceptionLabelMismatch.ExceptionDate] = this.ExceptionDate;
			this.ExceptionData[TABLE.ExceptionLabelMismatch.DivisionCode] = Common.LogonUser.LogonUserDivisionCode;
		}

		///<Developers>
		///	<Developer>Krzysztof Dobranowski</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>6/1/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="2255"> 
		///		<ExpectedInput>Valid exception data</ExpectedInput>
		///		<ExpectedOutput>Valid exception DataRow</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="2256"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		///    Generates exception report entry for an expired unit received. 
		/// </summary>
		/// <param name="bloodUnit" type="gov.va.med.vbecs.BOL.BloodUnit"></param>
		/// <returns>
		///     A void value...
		/// </returns>
		public void GenerateExceptionExpiredUnitReceived(BOL.BloodUnit bloodUnit)
		{
			DataTable dtExceptionExpiredUnitReceived = new DataTable(TABLE.ExceptionExpiredUnitReceived.TableName);
			dtExceptionExpiredUnitReceived.Columns.Add(TABLE.ExceptionExpiredUnitReceived.ExceptionExpiredUnitReceivedGuid, typeof(Guid));
			dtExceptionExpiredUnitReceived.Columns.Add(TABLE.ExceptionExpiredUnitReceived.ExceptionTypeId, typeof(int));
			dtExceptionExpiredUnitReceived.Columns.Add(TABLE.ExceptionExpiredUnitReceived.UnitId, typeof(string));
			dtExceptionExpiredUnitReceived.Columns.Add(TABLE.ExceptionExpiredUnitReceived.UnitGuid, typeof(Guid));
			dtExceptionExpiredUnitReceived.Columns.Add(TABLE.ExceptionExpiredUnitReceived.UnitAboRh, typeof(string));
			dtExceptionExpiredUnitReceived.Columns.Add(TABLE.ExceptionExpiredUnitReceived.ProductType, typeof(string));
			dtExceptionExpiredUnitReceived.Columns.Add(TABLE.ExceptionExpiredUnitReceived.ProductCode, typeof(string));
			dtExceptionExpiredUnitReceived.Columns.Add(TABLE.ExceptionExpiredUnitReceived.ExpirationDate, typeof(DateTime));
			dtExceptionExpiredUnitReceived.Columns.Add(TABLE.ExceptionExpiredUnitReceived.IncomingShipmentDate, typeof(DateTime));
			dtExceptionExpiredUnitReceived.Columns.Add(TABLE.ExceptionExpiredUnitReceived.ProcessingTechId, typeof(string));
			dtExceptionExpiredUnitReceived.Columns.Add(TABLE.ExceptionExpiredUnitReceived.ExceptionUser, typeof(string));
			dtExceptionExpiredUnitReceived.Columns.Add(TABLE.ExceptionExpiredUnitReceived.ExceptionComment, typeof(string));
			dtExceptionExpiredUnitReceived.Columns.Add(TABLE.ExceptionExpiredUnitReceived.ExceptionDate, typeof(DateTime));
			dtExceptionExpiredUnitReceived.Columns.Add(TABLE.ExceptionExpiredUnitReceived.DivisionCode, typeof(string));
			dtExceptionExpiredUnitReceived.Columns.Add(TABLE.ExceptionExpiredUnitReceived.LastUpdateUser, typeof(string));


			this.ExceptionData = dtExceptionExpiredUnitReceived.NewRow();
			this.ExceptionData[TABLE.ExceptionExpiredUnitReceived.ExceptionExpiredUnitReceivedGuid] = Guid.NewGuid();
			int nTest = (int) this.ExceptionType;
			this.ExceptionData[TABLE.ExceptionExpiredUnitReceived.ExceptionTypeId] = nTest;
			this.ExceptionData[TABLE.ExceptionExpiredUnitReceived.UnitId] = bloodUnit.EyeReadableUnitId;
			this.ExceptionData[TABLE.ExceptionExpiredUnitReceived.UnitGuid] = bloodUnit.BloodUnitGuid;
			this.ExceptionData[TABLE.ExceptionExpiredUnitReceived.UnitAboRh] = bloodUnit.BloodUnitMedia.AboRh.AboRHText;
			this.ExceptionData[TABLE.ExceptionExpiredUnitReceived.ProductType] = bloodUnit.ProductType.ProductTypeText;
			this.ExceptionData[TABLE.ExceptionExpiredUnitReceived.ProductCode] = bloodUnit.ProductCode;
			this.ExceptionData[TABLE.ExceptionExpiredUnitReceived.ExpirationDate] = bloodUnit.BloodUnitMedia.UnitExpirationDate;
			this.ExceptionData[TABLE.ExceptionExpiredUnitReceived.IncomingShipmentDate] = bloodUnit.IncomingShipment.ShipmentDate;
			this.ExceptionData[TABLE.ExceptionExpiredUnitReceived.ProcessingTechId] = bloodUnit.ProcessingTechId;
			this.ExceptionData[TABLE.ExceptionExpiredUnitReceived.ExceptionUser] = Common.LogonUser.LogonUserName;
			this.ExceptionData[TABLE.ExceptionExpiredUnitReceived.ExceptionComment] = bloodUnit.BloodUnitMedia.Comments;
			this.ExceptionData[TABLE.ExceptionExpiredUnitReceived.ExceptionDate] = BOL.VBECSDateTime.GetDivisionCurrentDateTime();
			this.ExceptionData[TABLE.ExceptionExpiredUnitReceived.DivisionCode] = Common.LogonUser.LogonUserDivisionCode;
			this.ExceptionData[TABLE.ExceptionExpiredUnitReceived.LastUpdateUser] = Common.LogonUser.LogonUserName;
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>05/31/2005</CreationDate>
		///<TestCases>
		///
		///<Case type="0" testid ="7461"> 
		///		<ExpectedInput>Valid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="7462"> 
		///		<ExpectedInput>Invalid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Generates exception for remove final unit status
		/// </summary>
		/// <param name="bloodUnit"></param>
		/// <param name="previousUnitStatus"></param>
		public void GenerateExceptionRemoveFinalUnitStatus(BOL.BloodUnit bloodUnit, Common.UnitStatusCode previousUnitStatus)
		{
			DataTable dtExceptionRemoveUnitStatus = new DataTable(TABLE.ExceptionRemoveFinalUnitStatus.TableName);
			dtExceptionRemoveUnitStatus.Columns.Add(TABLE.ExceptionRemoveFinalUnitStatus.ExceptionRemoveFinalUnitStatusGuid, typeof(Guid));
			dtExceptionRemoveUnitStatus.Columns.Add(TABLE.ExceptionRemoveFinalUnitStatus.ExceptionTypeId, typeof(int));
			dtExceptionRemoveUnitStatus.Columns.Add(TABLE.ExceptionRemoveFinalUnitStatus.UnitGuid, typeof(Guid));
			dtExceptionRemoveUnitStatus.Columns.Add(TABLE.ExceptionRemoveFinalUnitStatus.UnitId);
			dtExceptionRemoveUnitStatus.Columns.Add(TABLE.ExceptionRemoveFinalUnitStatus.UnitAboRh);
			dtExceptionRemoveUnitStatus.Columns.Add(TABLE.ExceptionRemoveFinalUnitStatus.CurrentUnitExpirationDate, typeof(DateTime));
			dtExceptionRemoveUnitStatus.Columns.Add(TABLE.ExceptionRemoveFinalUnitStatus.CurrentUnitProductType);
			dtExceptionRemoveUnitStatus.Columns.Add(TABLE.ExceptionRemoveFinalUnitStatus.CurrentUnitStatus);
			dtExceptionRemoveUnitStatus.Columns.Add(TABLE.ExceptionRemoveFinalUnitStatus.CurrentUnitVolume, typeof(decimal));			
			dtExceptionRemoveUnitStatus.Columns.Add(TABLE.ExceptionRemoveFinalUnitStatus.PreviousUnitExpirationDate, typeof(DateTime));
			dtExceptionRemoveUnitStatus.Columns.Add(TABLE.ExceptionRemoveFinalUnitStatus.PreviousUnitProductType);
			dtExceptionRemoveUnitStatus.Columns.Add(TABLE.ExceptionRemoveFinalUnitStatus.PreviousUnitStatus);
			dtExceptionRemoveUnitStatus.Columns.Add(TABLE.ExceptionRemoveFinalUnitStatus.PreviousUnitVolume, typeof(decimal));
			dtExceptionRemoveUnitStatus.Columns.Add(TABLE.ExceptionRemoveFinalUnitStatus.ExceptionUser);
			dtExceptionRemoveUnitStatus.Columns.Add(TABLE.ExceptionRemoveFinalUnitStatus.ExceptionComment);
			dtExceptionRemoveUnitStatus.Columns.Add(TABLE.ExceptionRemoveFinalUnitStatus.ExceptionDate, typeof(DateTime));
			dtExceptionRemoveUnitStatus.Columns.Add(TABLE.ExceptionRemoveFinalUnitStatus.DivisionCode);

			this.ExceptionData = dtExceptionRemoveUnitStatus.NewRow();
			this.ExceptionData[TABLE.ExceptionRemoveFinalUnitStatus.ExceptionRemoveFinalUnitStatusGuid] = Guid.NewGuid();
			this.ExceptionData[TABLE.ExceptionRemoveFinalUnitStatus.ExceptionTypeId] = this.ExceptionType;
			this.ExceptionData[TABLE.ExceptionRemoveFinalUnitStatus.UnitGuid] = bloodUnit.BloodUnitGuid;
			this.ExceptionData[TABLE.ExceptionRemoveFinalUnitStatus.UnitId] = bloodUnit.EyeReadableUnitId;
			this.ExceptionData[TABLE.ExceptionRemoveFinalUnitStatus.UnitAboRh] = bloodUnit.BloodUnitMedia.AboRh.AboRHText;
			this.ExceptionData[TABLE.ExceptionRemoveFinalUnitStatus.CurrentUnitExpirationDate] = bloodUnit.BloodUnitMedia.UnitExpirationDate;
			this.ExceptionData[TABLE.ExceptionRemoveFinalUnitStatus.CurrentUnitProductType] = bloodUnit.ProductShortName;
			this.ExceptionData[TABLE.ExceptionRemoveFinalUnitStatus.CurrentUnitStatus] = Common.Utility.GetUnitStatusTextFromEnum(bloodUnit.BloodUnitStatus.UnitStatusCode);
			this.ExceptionData[TABLE.ExceptionRemoveFinalUnitStatus.CurrentUnitVolume] = bloodUnit.OriginalVolume;
			this.ExceptionData[TABLE.ExceptionRemoveFinalUnitStatus.PreviousUnitExpirationDate] = bloodUnit.BloodUnitMedia.UnitExpirationDate;
			this.ExceptionData[TABLE.ExceptionRemoveFinalUnitStatus.PreviousUnitProductType] = bloodUnit.ProductShortName;
			this.ExceptionData[TABLE.ExceptionRemoveFinalUnitStatus.PreviousUnitStatus] = Common.Utility.GetUnitStatusTextFromEnum(previousUnitStatus);
			this.ExceptionData[TABLE.ExceptionRemoveFinalUnitStatus.PreviousUnitVolume] = bloodUnit.OriginalVolume;
			this.ExceptionData[TABLE.ExceptionRemoveFinalUnitStatus.ExceptionUser] = this.ExceptionUser;
			this.ExceptionData[TABLE.ExceptionRemoveFinalUnitStatus.ExceptionComment] = this.ExceptionComment;
			this.ExceptionData[TABLE.ExceptionRemoveFinalUnitStatus.ExceptionDate] = this.ExceptionDate;
			this.ExceptionData[TABLE.ExceptionRemoveFinalUnitStatus.DivisionCode] = Common.LogonUser.LogonUserDivisionCode;
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/8/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="7463"> 
		///		<ExpectedInput>Valid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="7464"> 
		///		<ExpectedInput>Invalid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Generates exception for remove unit modification
		/// </summary>
		/// <param name="bloodUnit"></param>
		/// <param name="bloodUnitModification"></param>
		public void GenerateExceptionRemoveUnitModification(BOL.BloodUnit bloodUnit, DataRow bloodUnitModification)
		{
			DataTable dtExceptionRemoveUnitStatus = new DataTable(TABLE.ExceptionRemoveFinalUnitStatus.TableName);
			dtExceptionRemoveUnitStatus.Columns.Add(TABLE.ExceptionRemoveFinalUnitStatus.ExceptionRemoveFinalUnitStatusGuid, typeof(Guid));
			dtExceptionRemoveUnitStatus.Columns.Add(TABLE.ExceptionRemoveFinalUnitStatus.ExceptionTypeId, typeof(int));
			dtExceptionRemoveUnitStatus.Columns.Add(TABLE.ExceptionRemoveFinalUnitStatus.UnitGuid, typeof(Guid));
			dtExceptionRemoveUnitStatus.Columns.Add(TABLE.ExceptionRemoveFinalUnitStatus.UnitId);
			dtExceptionRemoveUnitStatus.Columns.Add(TABLE.ExceptionRemoveFinalUnitStatus.UnitAboRh);
			dtExceptionRemoveUnitStatus.Columns.Add(TABLE.ExceptionRemoveFinalUnitStatus.CurrentUnitExpirationDate, typeof(DateTime));
			dtExceptionRemoveUnitStatus.Columns.Add(TABLE.ExceptionRemoveFinalUnitStatus.CurrentUnitProductType);
			dtExceptionRemoveUnitStatus.Columns.Add(TABLE.ExceptionRemoveFinalUnitStatus.CurrentUnitStatus);
			dtExceptionRemoveUnitStatus.Columns.Add(TABLE.ExceptionRemoveFinalUnitStatus.CurrentUnitVolume, typeof(decimal));			
			dtExceptionRemoveUnitStatus.Columns.Add(TABLE.ExceptionRemoveFinalUnitStatus.PreviousUnitExpirationDate, typeof(DateTime));
			dtExceptionRemoveUnitStatus.Columns.Add(TABLE.ExceptionRemoveFinalUnitStatus.PreviousUnitProductType);
			dtExceptionRemoveUnitStatus.Columns.Add(TABLE.ExceptionRemoveFinalUnitStatus.PreviousUnitStatus);
			dtExceptionRemoveUnitStatus.Columns.Add(TABLE.ExceptionRemoveFinalUnitStatus.PreviousUnitVolume, typeof(decimal));
			dtExceptionRemoveUnitStatus.Columns.Add(TABLE.ExceptionRemoveFinalUnitStatus.ExceptionUser);
			dtExceptionRemoveUnitStatus.Columns.Add(TABLE.ExceptionRemoveFinalUnitStatus.ExceptionComment);
			dtExceptionRemoveUnitStatus.Columns.Add(TABLE.ExceptionRemoveFinalUnitStatus.ExceptionDate, typeof(DateTime));
			dtExceptionRemoveUnitStatus.Columns.Add(TABLE.ExceptionRemoveFinalUnitStatus.DivisionCode);

			this.ExceptionData = dtExceptionRemoveUnitStatus.NewRow();
			this.ExceptionData[TABLE.ExceptionRemoveFinalUnitStatus.ExceptionRemoveFinalUnitStatusGuid] = Guid.NewGuid();
			this.ExceptionData[TABLE.ExceptionRemoveFinalUnitStatus.ExceptionTypeId] = this.ExceptionType;
			this.ExceptionData[TABLE.ExceptionRemoveFinalUnitStatus.UnitGuid] = bloodUnit.BloodUnitGuid;
			this.ExceptionData[TABLE.ExceptionRemoveFinalUnitStatus.UnitId] = bloodUnit.EyeReadableUnitId;
			this.ExceptionData[TABLE.ExceptionRemoveFinalUnitStatus.UnitAboRh] = bloodUnit.BloodUnitMedia.AboRh.AboRHText;
			this.ExceptionData[TABLE.ExceptionRemoveFinalUnitStatus.CurrentUnitExpirationDate] = bloodUnit.BloodUnitMedia.UnitExpirationDate;
			this.ExceptionData[TABLE.ExceptionRemoveFinalUnitStatus.CurrentUnitProductType] = bloodUnit.ProductShortName;
			this.ExceptionData[TABLE.ExceptionRemoveFinalUnitStatus.CurrentUnitStatus] = Common.Utility.GetUnitStatusTextFromEnum(bloodUnit.BloodUnitStatus.UnitStatusCode);
			this.ExceptionData[TABLE.ExceptionRemoveFinalUnitStatus.CurrentUnitVolume] = bloodUnit.OriginalVolume;
			this.ExceptionData[TABLE.ExceptionRemoveFinalUnitStatus.PreviousUnitExpirationDate] = bloodUnitModification[TABLE.BloodUnitModification.FromUnitExpirationDate];
			BOL.BloodProduct fromProduct = new BOL.BloodProduct((Guid)bloodUnitModification[TABLE.BloodUnitModification.FromBloodProductGuid]);
			this.ExceptionData[TABLE.ExceptionRemoveFinalUnitStatus.PreviousUnitProductType] = fromProduct.ProductShortName;
			this.ExceptionData[TABLE.ExceptionRemoveFinalUnitStatus.PreviousUnitStatus] = Common.Utility.GetUnitStatusTextFromEnum(bloodUnit.BloodUnitStatus.UnitStatusCode);
			this.ExceptionData[TABLE.ExceptionRemoveFinalUnitStatus.PreviousUnitVolume] = bloodUnitModification[TABLE.BloodUnitModification.FromOriginalVolume];
			this.ExceptionData[TABLE.ExceptionRemoveFinalUnitStatus.ExceptionUser] = this.ExceptionUser;
			this.ExceptionData[TABLE.ExceptionRemoveFinalUnitStatus.ExceptionComment] = this.ExceptionComment;
			this.ExceptionData[TABLE.ExceptionRemoveFinalUnitStatus.ExceptionDate] = this.ExceptionDate;
			this.ExceptionData[TABLE.ExceptionRemoveFinalUnitStatus.DivisionCode] = Common.LogonUser.LogonUserDivisionCode;
		}

		///<Developers>
		///	<Developer>Margaret Jablonski</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/6/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4832"> 
		///		<ExpectedInput>Valid exception data</ExpectedInput>
		///		<ExpectedOutput>Exception Data</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4833"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Generates exception for return issued unit
		/// </summary>
		/// <param name="issuedUnit"></param>
		/// <param name="bloodUnit"></param>
		/// <param name="patient"></param>
		public void GenerateExceptionReturnIssuedUnit(BOL.IssuedUnit issuedUnit, BOL.Patient patient, BOL.BloodUnit bloodUnit)
		{
			DataTable dtExceptionReturnIssuedUnit = new DataTable(TABLE.ExceptionReturnedUnit.TableName);
			dtExceptionReturnIssuedUnit.Columns.Add(TABLE.ExceptionReturnedUnit.ExceptionReturnedUnitGuid, typeof(Guid));
			dtExceptionReturnIssuedUnit.Columns.Add(TABLE.ExceptionReturnedUnit.ExceptionTypeId, typeof(int));
			dtExceptionReturnIssuedUnit.Columns.Add(TABLE.ExceptionReturnedUnit.IssueToLocation);
			dtExceptionReturnIssuedUnit.Columns.Add(TABLE.ExceptionReturnedUnit.PatientGuid, typeof(Guid));
			dtExceptionReturnIssuedUnit.Columns.Add(TABLE.ExceptionReturnedUnit.PatientAboRh);
			dtExceptionReturnIssuedUnit.Columns.Add(TABLE.ExceptionReturnedUnit.PatientId);
			dtExceptionReturnIssuedUnit.Columns.Add(TABLE.ExceptionReturnedUnit.PatientName);
			dtExceptionReturnIssuedUnit.Columns.Add(TABLE.ExceptionReturnedUnit.ProcessingTechName);
			dtExceptionReturnIssuedUnit.Columns.Add(TABLE.ExceptionReturnedUnit.ReturnedUnitAboRh);
			dtExceptionReturnIssuedUnit.Columns.Add(TABLE.ExceptionReturnedUnit.ReturnedUnitExpirationDate, typeof(DateTime));
			dtExceptionReturnIssuedUnit.Columns.Add(TABLE.ExceptionReturnedUnit.ReturnedUnitId);
			dtExceptionReturnIssuedUnit.Columns.Add(TABLE.ExceptionReturnedUnit.ReturnedUnitGuid);
			dtExceptionReturnIssuedUnit.Columns.Add(TABLE.ExceptionReturnedUnit.ReturnedUnitProductShortName);
			dtExceptionReturnIssuedUnit.Columns.Add(TABLE.ExceptionReturnedUnit.VisualInspectionIndicator, typeof(bool));
			dtExceptionReturnIssuedUnit.Columns.Add(TABLE.ExceptionReturnedUnit.ExceptionUser);
			dtExceptionReturnIssuedUnit.Columns.Add(TABLE.ExceptionReturnedUnit.ExceptionComment);
			dtExceptionReturnIssuedUnit.Columns.Add(TABLE.ExceptionReturnedUnit.ExceptionDate, typeof(DateTime));
			dtExceptionReturnIssuedUnit.Columns.Add(TABLE.ExceptionReturnedUnit.DivisionCode);

			this.ExceptionData = dtExceptionReturnIssuedUnit.NewRow();
			this.ExceptionData[TABLE.ExceptionReturnedUnit.ExceptionReturnedUnitGuid] = Guid.NewGuid();
			this.ExceptionData[TABLE.ExceptionReturnedUnit.ExceptionTypeId] = this.ExceptionType;
			this.ExceptionData[TABLE.ExceptionReturnedUnit.IssueToLocation] = issuedUnit.IssueToLocationName;
			this.ExceptionData[TABLE.ExceptionReturnedUnit.PatientGuid] = patient.PatientGuid;
			this.ExceptionData[TABLE.ExceptionReturnedUnit.PatientAboRh] = patient.AboRh.AboRHText;
			this.ExceptionData[TABLE.ExceptionReturnedUnit.PatientId] = patient.DisplayPatientID;
			this.ExceptionData[TABLE.ExceptionReturnedUnit.PatientName] = patient.DisplayName;
			this.ExceptionData[TABLE.ExceptionReturnedUnit.ProcessingTechName] = issuedUnit.ReturnTechId;
			this.ExceptionData[TABLE.ExceptionReturnedUnit.ReturnedUnitAboRh] = bloodUnit.BloodUnitMedia.AboRh.AboRHText;
			this.ExceptionData[TABLE.ExceptionReturnedUnit.ReturnedUnitExpirationDate] = bloodUnit.BloodUnitMedia.UnitExpirationDate;
			this.ExceptionData[TABLE.ExceptionReturnedUnit.ReturnedUnitId] = bloodUnit.EyeReadableUnitId;
			this.ExceptionData[TABLE.ExceptionReturnedUnit.ReturnedUnitGuid] = bloodUnit.BloodUnitGuid;
			this.ExceptionData[TABLE.ExceptionReturnedUnit.ReturnedUnitProductShortName] = bloodUnit.ProductShortName;
			this.ExceptionData[TABLE.ExceptionReturnedUnit.VisualInspectionIndicator] = issuedUnit.ReturnSatisfactoryIndicator;
			this.ExceptionData[TABLE.ExceptionReturnedUnit.ExceptionUser] = this.ExceptionUser;
			this.ExceptionData[TABLE.ExceptionReturnedUnit.ExceptionComment] = this.ExceptionComment;
			this.ExceptionData[TABLE.ExceptionReturnedUnit.ExceptionDate] = this.ExceptionDate;
			this.ExceptionData[TABLE.ExceptionReturnedUnit.DivisionCode] = Common.LogonUser.LogonUserDivisionCode;
		}


		///<Developers>
		///	<Developer>Margaret Jablonski</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/6/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4830"> 
		///		<ExpectedInput>Valid exception data</ExpectedInput>
		///		<ExpectedOutput>Exception Data</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4831"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Generates exception for Abo Incompatible Transfusion
		/// </summary>
		/// <param name="bloodUnit"></param>
		/// <param name="patient"></param>
		public void GenerateExceptionAboIncompatibleTransfusion(BOL.Patient patient, BOL.BloodUnit bloodUnit)
		{
			DataTable dtExceptionTransfusion = new DataTable(TABLE.ExceptionTransfusion.TableName);
			dtExceptionTransfusion.Columns.Add(TABLE.ExceptionTransfusion.ExceptionTransfusionGuid, typeof(Guid));
			dtExceptionTransfusion.Columns.Add(TABLE.ExceptionTransfusion.ExceptionTypeId, typeof(int));
			dtExceptionTransfusion.Columns.Add(TABLE.ExceptionTransfusion.UnitId);
			dtExceptionTransfusion.Columns.Add(TABLE.ExceptionTransfusion.UnitGuid);
			dtExceptionTransfusion.Columns.Add(TABLE.ExceptionTransfusion.UnitProductType);
			dtExceptionTransfusion.Columns.Add(TABLE.ExceptionTransfusion.UnitAboRh);
			dtExceptionTransfusion.Columns.Add(TABLE.ExceptionTransfusion.UnitExpirationDate, typeof(DateTime));
			dtExceptionTransfusion.Columns.Add(TABLE.ExceptionTransfusion.TransfusedPatientGuid,typeof(Guid));
			dtExceptionTransfusion.Columns.Add(TABLE.ExceptionTransfusion.TransfusedPatientName);
			dtExceptionTransfusion.Columns.Add(TABLE.ExceptionTransfusion.TransfusedPatientId);
			dtExceptionTransfusion.Columns.Add(TABLE.ExceptionTransfusion.TransfusedPatientAboRh);
			dtExceptionTransfusion.Columns.Add(TABLE.ExceptionTransfusion.ExceptionUser);
			dtExceptionTransfusion.Columns.Add(TABLE.ExceptionTransfusion.ExceptionComment);
			dtExceptionTransfusion.Columns.Add(TABLE.ExceptionTransfusion.ExceptionDate, typeof(DateTime));
			dtExceptionTransfusion.Columns.Add(TABLE.ExceptionTransfusion.DivisionCode);

			this.ExceptionData = dtExceptionTransfusion.NewRow();
			this.ExceptionData[TABLE.ExceptionTransfusion.ExceptionTransfusionGuid] = Guid.NewGuid();
			this.ExceptionData[TABLE.ExceptionTransfusion.ExceptionTypeId] = this.ExceptionType;
			this.ExceptionData[TABLE.ExceptionTransfusion.UnitId] = bloodUnit.EyeReadableUnitId;
			this.ExceptionData[TABLE.ExceptionTransfusion.UnitGuid] = bloodUnit.BloodUnitGuid;
			this.ExceptionData[TABLE.ExceptionTransfusion.UnitProductType] = bloodUnit.ProductType.ProductTypeText;
			this.ExceptionData[TABLE.ExceptionTransfusion.UnitAboRh] = bloodUnit.BloodUnitMedia.AboRh.AboRHText;
			this.ExceptionData[TABLE.ExceptionTransfusion.UnitExpirationDate] = bloodUnit.BloodUnitMedia.UnitExpirationDate;
			this.ExceptionData[TABLE.ExceptionTransfusion.TransfusedPatientGuid] = patient.PatientGuid;
			this.ExceptionData[TABLE.ExceptionTransfusion.TransfusedPatientName] = patient.DisplayName;
			this.ExceptionData[TABLE.ExceptionTransfusion.TransfusedPatientId] = patient.DisplayPatientID;
			this.ExceptionData[TABLE.ExceptionTransfusion.TransfusedPatientAboRh] = patient.AboRh.AboRHText;
			this.ExceptionData[TABLE.ExceptionTransfusion.ExceptionUser] = this.ExceptionUser;
			this.ExceptionData[TABLE.ExceptionTransfusion.ExceptionComment] = this.ExceptionComment;
			this.ExceptionData[TABLE.ExceptionTransfusion.ExceptionDate] = this.ExceptionDate;
			this.ExceptionData[TABLE.ExceptionTransfusion.DivisionCode] = Common.LogonUser.LogonUserDivisionCode;
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/26/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="2257"> 
		///		<ExpectedInput>Valid exception data</ExpectedInput>
		///		<ExpectedOutput>Exception Data</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="2262"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// GenerateExceptionTrwUnitClericalCheckFailed
		/// </summary>
		/// <param name="patientTransfusionReaction">Patient Transfusion Reaction object</param>
		/// <param name="patient">Patient object</param>
		/// <param name="bloodUnit">Blood Unit object</param>
		/// <param name="transfusionReaction">Transfusion reaction object</param>
		public void GenerateExceptionTrwUnitClericalCheckFailed(BOL.PatientTransfusionReaction patientTransfusionReaction, BOL.Patient patient, BOL.BloodUnit bloodUnit, BOL.TransfusionReaction transfusionReaction)
		{
			BOL.IssuedUnit issuedUnit = BOL.IssuedUnit.GetIssuedUnitFromBloodUnitGuid(bloodUnit.BloodUnitGuid);
			//
			DataTable dtException = new DataTable(TABLE.ExceptionTransfusionReactionWorkup.TableName);
			dtException.Columns.Add(TABLE.ExceptionTransfusionReactionWorkup.ExceptionTransfusionReactionWorkupGuid, typeof(Guid));
			dtException.Columns.Add(TABLE.ExceptionTransfusionReactionWorkup.ExceptionTypeId, typeof(int));
			dtException.Columns.Add(TABLE.ExceptionTransfusionReactionWorkup.UnitId);
			dtException.Columns.Add(TABLE.ExceptionTransfusionReactionWorkup.UnitGuid);
			dtException.Columns.Add(TABLE.ExceptionTransfusionReactionWorkup.UnitAboRh);
			dtException.Columns.Add(TABLE.ExceptionTransfusionReactionWorkup.ProductShortName);
			dtException.Columns.Add(TABLE.ExceptionTransfusionReactionWorkup.ProductCode);
			dtException.Columns.Add(TABLE.ExceptionTransfusionReactionWorkup.UnitExpirationDate, typeof(DateTime));
			dtException.Columns.Add(TABLE.ExceptionTransfusionReactionWorkup.UnitDonationType);
			dtException.Columns.Add(TABLE.ExceptionTransfusionReactionWorkup.TrwPatientGuid, typeof(Guid));
			dtException.Columns.Add(TABLE.ExceptionTransfusionReactionWorkup.TrwPatientName);
			dtException.Columns.Add(TABLE.ExceptionTransfusionReactionWorkup.TrwPatientId);
			dtException.Columns.Add(TABLE.ExceptionTransfusionReactionWorkup.AssignedPatientGuid, typeof(Guid));
			dtException.Columns.Add(TABLE.ExceptionTransfusionReactionWorkup.AssignedPatientName);
			dtException.Columns.Add(TABLE.ExceptionTransfusionReactionWorkup.AssignedPatientId);
			dtException.Columns.Add(TABLE.ExceptionTransfusionReactionWorkup.ClericalCheckFailed, typeof(bool));
			dtException.Columns.Add(TABLE.ExceptionTransfusionReactionWorkup.IssueDate, typeof(DateTime));
			dtException.Columns.Add(TABLE.ExceptionTransfusionReactionWorkup.IssueTechId);
			dtException.Columns.Add(TABLE.ExceptionTransfusionReactionWorkup.TrwDate, typeof(DateTime));
			dtException.Columns.Add(TABLE.ExceptionTransfusionReactionWorkup.TrwTechId);
			dtException.Columns.Add(TABLE.ExceptionTransfusionReactionWorkup.ExceptionUser);
			dtException.Columns.Add(TABLE.ExceptionTransfusionReactionWorkup.ExceptionComment);
			dtException.Columns.Add(TABLE.ExceptionTransfusionReactionWorkup.ExceptionDate, typeof(DateTime));
			dtException.Columns.Add(TABLE.ExceptionTransfusionReactionWorkup.DivisionCode);
			//
			this.ExceptionData = dtException.NewRow();
			this.ExceptionData[TABLE.ExceptionTransfusionReactionWorkup.ExceptionTransfusionReactionWorkupGuid] = Guid.NewGuid();
			this.ExceptionData[TABLE.ExceptionTransfusionReactionWorkup.ExceptionTypeId] = (int)this.ExceptionType;
			this.ExceptionData[TABLE.ExceptionTransfusionReactionWorkup.UnitId] = bloodUnit.EyeReadableUnitId;
			this.ExceptionData[TABLE.ExceptionTransfusionReactionWorkup.UnitGuid] = bloodUnit.BloodUnitGuid;
			this.ExceptionData[TABLE.ExceptionTransfusionReactionWorkup.UnitAboRh] = bloodUnit.BloodUnitMedia.AboRh.AboRHText;
			this.ExceptionData[TABLE.ExceptionTransfusionReactionWorkup.ProductShortName] = bloodUnit.ProductShortName;
			this.ExceptionData[TABLE.ExceptionTransfusionReactionWorkup.ProductCode] = bloodUnit.ProductCode;
			this.ExceptionData[TABLE.ExceptionTransfusionReactionWorkup.UnitExpirationDate] = bloodUnit.BloodUnitMedia.UnitExpirationDate;
			this.ExceptionData[TABLE.ExceptionTransfusionReactionWorkup.UnitDonationType] = bloodUnit.DonationType.DonationTypeText;
			this.ExceptionData[TABLE.ExceptionTransfusionReactionWorkup.TrwPatientGuid] = patient.PatientGuid;
			this.ExceptionData[TABLE.ExceptionTransfusionReactionWorkup.TrwPatientName] = patient.DisplayName;
			this.ExceptionData[TABLE.ExceptionTransfusionReactionWorkup.TrwPatientId] = patient.DisplayPatientID;
			this.ExceptionData[TABLE.ExceptionTransfusionReactionWorkup.AssignedPatientGuid] = patient.PatientGuid;
			this.ExceptionData[TABLE.ExceptionTransfusionReactionWorkup.AssignedPatientName] = patient.DisplayName;
			this.ExceptionData[TABLE.ExceptionTransfusionReactionWorkup.AssignedPatientId] = patient.DisplayPatientID;
			this.ExceptionData[TABLE.ExceptionTransfusionReactionWorkup.ClericalCheckFailed] = !transfusionReaction.BagClericalChecksOkIndicator;
			this.ExceptionData[TABLE.ExceptionTransfusionReactionWorkup.IssueDate] = issuedUnit.IssueDate;
			this.ExceptionData[TABLE.ExceptionTransfusionReactionWorkup.IssueTechId] = GetUserNameFromID(issuedUnit.IssueTechId);
			this.ExceptionData[TABLE.ExceptionTransfusionReactionWorkup.TrwDate] = patientTransfusionReaction.InvestigatedDateTime;
			this.ExceptionData[TABLE.ExceptionTransfusionReactionWorkup.TrwTechId] = GetUserNameFromID(patientTransfusionReaction.EntryTechId);
			this.ExceptionData[TABLE.ExceptionTransfusionReactionWorkup.ExceptionUser] = this.ExceptionUser;
			this.ExceptionData[TABLE.ExceptionTransfusionReactionWorkup.ExceptionComment] = this.ExceptionComment;
			this.ExceptionData[TABLE.ExceptionTransfusionReactionWorkup.ExceptionDate] = this.ExceptionDate;
			this.ExceptionData[TABLE.ExceptionTransfusionReactionWorkup.DivisionCode] = Common.LogonUser.LogonUserDivisionCode;
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/26/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="2271"> 
		///		<ExpectedInput>Valid exception data</ExpectedInput>
		///		<ExpectedOutput>Exception Data</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="2279"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// GenerateExceptionTrwSpecimen
		/// </summary>
		/// <param name="patientTransfusionReaction">Patient Transfusion Reaction object</param>
		/// <param name="patient">Patient object</param>
		/// <param name="specimen">Specimen object</param>
		/// <param name="clericalCheckOK">Clerical Check OK indicator</param>
		public void GenerateExceptionTrwSpecimen(BOL.PatientTransfusionReaction patientTransfusionReaction, BOL.Patient patient, BOL.Specimen specimen, bool clericalCheckOK)
		{
			DataTable dtException = new DataTable(TABLE.ExceptionTrwSpecimen.TableName);
			dtException.Columns.Add(TABLE.ExceptionTrwSpecimen.ExceptionTrwSpecimenGuid, typeof(Guid));
			dtException.Columns.Add(TABLE.ExceptionTrwSpecimen.ExceptionTypeId, typeof(int));
			dtException.Columns.Add(TABLE.ExceptionTrwSpecimen.TrwPatientGuid, typeof(Guid));
			dtException.Columns.Add(TABLE.ExceptionTrwSpecimen.TrwPatientName);
			dtException.Columns.Add(TABLE.ExceptionTrwSpecimen.TrwPatientId);
			dtException.Columns.Add(TABLE.ExceptionTrwSpecimen.SpecimenUid);
			dtException.Columns.Add(TABLE.ExceptionTrwSpecimen.SpecimenPatientGuid, typeof(Guid));
			dtException.Columns.Add(TABLE.ExceptionTrwSpecimen.SpecimenPatientName);
			dtException.Columns.Add(TABLE.ExceptionTrwSpecimen.SpecimenPatientId);
			dtException.Columns.Add(TABLE.ExceptionTrwSpecimen.ClericalCheckFailed, typeof(bool));
			dtException.Columns.Add(TABLE.ExceptionTrwSpecimen.TrwDate, typeof(DateTime));
			dtException.Columns.Add(TABLE.ExceptionTrwSpecimen.TrwTechId);
			dtException.Columns.Add(TABLE.ExceptionTrwSpecimen.ExceptionUser);
			dtException.Columns.Add(TABLE.ExceptionTrwSpecimen.ExceptionComment);
			dtException.Columns.Add(TABLE.ExceptionTrwSpecimen.ExceptionDate, typeof(DateTime));
			dtException.Columns.Add(TABLE.ExceptionTrwSpecimen.DivisionCode);
			//
			this.ExceptionData = dtException.NewRow();
			this.ExceptionData[TABLE.ExceptionTrwSpecimen.ExceptionTrwSpecimenGuid] = Guid.NewGuid();
			this.ExceptionData[TABLE.ExceptionTrwSpecimen.ExceptionTypeId] = (int)this.ExceptionType;
			this.ExceptionData[TABLE.ExceptionTrwSpecimen.TrwPatientGuid] = patient.PatientGuid;
			this.ExceptionData[TABLE.ExceptionTrwSpecimen.TrwPatientName] = patient.DisplayName;
			this.ExceptionData[TABLE.ExceptionTrwSpecimen.TrwPatientId] = patient.DisplayPatientID;
			this.ExceptionData[TABLE.ExceptionTrwSpecimen.SpecimenUid] = specimen.SpecimenUid;
			this.ExceptionData[TABLE.ExceptionTrwSpecimen.SpecimenPatientGuid] = specimen.Patient.PatientGuid;
			this.ExceptionData[TABLE.ExceptionTrwSpecimen.SpecimenPatientName] = specimen.Patient.DisplayName;
			this.ExceptionData[TABLE.ExceptionTrwSpecimen.SpecimenPatientId] = specimen.Patient.DisplayPatientID;
			this.ExceptionData[TABLE.ExceptionTrwSpecimen.ClericalCheckFailed] = !clericalCheckOK;
			this.ExceptionData[TABLE.ExceptionTrwSpecimen.TrwDate] = patientTransfusionReaction.InvestigatedDateTime;
			this.ExceptionData[TABLE.ExceptionTrwSpecimen.TrwTechId] = GetUserNameFromID(patientTransfusionReaction.EntryTechId);
			this.ExceptionData[TABLE.ExceptionTrwSpecimen.ExceptionUser] = this.ExceptionUser;
			this.ExceptionData[TABLE.ExceptionTrwSpecimen.ExceptionComment] = this.ExceptionComment;
			this.ExceptionData[TABLE.ExceptionTrwSpecimen.ExceptionDate] = this.ExceptionDate;
			this.ExceptionData[TABLE.ExceptionTrwSpecimen.DivisionCode] = Common.LogonUser.LogonUserDivisionCode;
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>11/26/2004</CreationDate>
		///<TestCases>
		/// 
		///<Case type="0" testid ="7465"> 
		///		<ExpectedInput>Valid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="7466"> 
		///		<ExpectedInput>Invalid Parameters</ExpectedInput>
		///		<ExpectedOutput>NullReferenceException</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Generates exception for inactivated antigen test
		/// </summary>
		/// <param name="bloodUnitTest"></param>
		/// <param name="testingTechName"></param>
		/// <param name="testingDateTime"></param>
		public void GenerateExceptionAntigenTest(DataRow bloodUnitTest, string testingTechName, DateTime testingDateTime)
		{
			DataTable dtExceptionAntigenTest = new DataTable(TABLE.ExceptionAntigenTest.TableName);
			dtExceptionAntigenTest.Columns.Add(TABLE.ExceptionAntigenTest.ExceptionAntigenTestGuid, typeof(Guid));
			dtExceptionAntigenTest.Columns.Add(TABLE.ExceptionAntigenTest.ExceptionTypeId, typeof(int));
			dtExceptionAntigenTest.Columns.Add(TABLE.ExceptionAntigenTest.UnitGuid, typeof(Guid));
			dtExceptionAntigenTest.Columns.Add(TABLE.ExceptionAntigenTest.UnitId);
			dtExceptionAntigenTest.Columns.Add(TABLE.ExceptionAntigenTest.ProductShortName);
			dtExceptionAntigenTest.Columns.Add(TABLE.ExceptionAntigenTest.TestPerformed);
			dtExceptionAntigenTest.Columns.Add(TABLE.ExceptionAntigenTest.InvalidatedResult);
			dtExceptionAntigenTest.Columns.Add(TABLE.ExceptionAntigenTest.InvalidatedDate, typeof(DateTime));
			dtExceptionAntigenTest.Columns.Add(TABLE.ExceptionAntigenTest.PreviousTestingTechName);
			dtExceptionAntigenTest.Columns.Add(TABLE.ExceptionAntigenTest.CurrentTestingTechName);
			dtExceptionAntigenTest.Columns.Add(TABLE.ExceptionAntigenTest.ExceptionUser);
			dtExceptionAntigenTest.Columns.Add(TABLE.ExceptionAntigenTest.ExceptionComment);
			dtExceptionAntigenTest.Columns.Add(TABLE.ExceptionAntigenTest.ExceptionDate, typeof(DateTime));
			dtExceptionAntigenTest.Columns.Add(TABLE.ExceptionAntigenTest.DivisionCode);

			this.ExceptionData = dtExceptionAntigenTest.NewRow();
			this.ExceptionData[TABLE.ExceptionAntigenTest.ExceptionAntigenTestGuid] = Guid.NewGuid();
			this.ExceptionData[TABLE.ExceptionAntigenTest.ExceptionTypeId] = this.ExceptionType;
			this.ExceptionData[TABLE.ExceptionAntigenTest.UnitGuid] = (Guid)bloodUnitTest[TABLE.BloodUnitTest.BloodUnitGuid];
			this.ExceptionData[TABLE.ExceptionAntigenTest.UnitId] = bloodUnitTest[TABLE.BloodUnit.EyeReadableUnitId].ToString();
			this.ExceptionData[TABLE.ExceptionAntigenTest.ProductShortName] = bloodUnitTest[TABLE.BloodProduct.ProductShortName].ToString();
			this.ExceptionData[TABLE.ExceptionAntigenTest.TestPerformed] = bloodUnitTest[TABLE.OrderableTest.OrderableTestText].ToString();
			this.ExceptionData[TABLE.ExceptionAntigenTest.InvalidatedResult] = bloodUnitTest[TABLE.TestResult.TestResultText].ToString();
			this.ExceptionData[TABLE.ExceptionAntigenTest.InvalidatedDate] = Convert.ToDateTime(bloodUnitTest[TABLE.BloodUnitTest.TestDate]);
			this.ExceptionData[TABLE.ExceptionAntigenTest.PreviousTestingTechName] = GetUserNameFromID(bloodUnitTest[TABLE.BloodUnitTest.TestTechId].ToString());
			this.ExceptionData[TABLE.ExceptionAntigenTest.CurrentTestingTechName] = GetUserNameFromID(testingTechName);
			this.ExceptionData[TABLE.ExceptionAntigenTest.ExceptionUser] = this.ExceptionUser;
			this.ExceptionData[TABLE.ExceptionAntigenTest.ExceptionComment] = this.ExceptionComment;
			this.ExceptionData[TABLE.ExceptionAntigenTest.ExceptionDate] = this.ExceptionDate;
			this.ExceptionData[TABLE.ExceptionAntigenTest.DivisionCode] = Common.LogonUser.LogonUserDivisionCode;
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>11/29/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="7467"> 
		///		<ExpectedInput>Valid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="7468"> 
		///		<ExpectedInput>Invalid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Generates exception for unit testing phase change
		/// BR_41.26
		/// </summary>
		/// <param name="bloodUnitGuid"></param>
		/// <param name="bloodUnitId"></param>
		/// <param name="testName"></param>
		/// <param name="phaseOfReaction"></param>
		/// <param name="testInterp"></param>
		/// <param name="testDate"></param>
		/// <param name="testingTech"></param>
		public void GenerateExceptionUnitTest(Guid bloodUnitGuid, string bloodUnitId, string testName, string phaseOfReaction, string testInterp, DateTime testDate, string testingTech)
		{
			DataTable dtExceptionUnitTest = new DataTable("ExceptionUnitTest");
			dtExceptionUnitTest.Columns.Add(TABLE.ExceptionPatientTest.ExceptionPatientTestGuid, typeof(Guid));
			dtExceptionUnitTest.Columns.Add(TABLE.ExceptionPatientTest.ExceptionTypeId, typeof(int));
			dtExceptionUnitTest.Columns.Add(TABLE.ExceptionPatientTest.PatientGuid, typeof(Guid));
			dtExceptionUnitTest.Columns.Add(TABLE.ExceptionPatientTest.PatientName);
			dtExceptionUnitTest.Columns.Add(TABLE.ExceptionPatientTest.PatientId);
			dtExceptionUnitTest.Columns.Add(TABLE.ExceptionPatientTest.SpecimenUid);
			dtExceptionUnitTest.Columns.Add(TABLE.ExceptionPatientTest.UnitGuid, typeof(Guid));
			dtExceptionUnitTest.Columns.Add(TABLE.ExceptionPatientTest.UnitId);
			dtExceptionUnitTest.Columns.Add(TABLE.ExceptionPatientTest.TestName);
			dtExceptionUnitTest.Columns.Add(TABLE.ExceptionPatientTest.TestInterp);
			dtExceptionUnitTest.Columns.Add(TABLE.ExceptionPatientTest.TestDate);
			dtExceptionUnitTest.Columns.Add(TABLE.ExceptionPatientTest.TestingTechName);
			dtExceptionUnitTest.Columns.Add(TABLE.ExceptionPatientTest.ReactivityPhase);
			dtExceptionUnitTest.Columns.Add(TABLE.ExceptionPatientTest.ExceptionUser);
			dtExceptionUnitTest.Columns.Add(TABLE.ExceptionPatientTest.ExceptionComment);
			dtExceptionUnitTest.Columns.Add(TABLE.ExceptionPatientTest.ExceptionDate, typeof(DateTime));
			dtExceptionUnitTest.Columns.Add(TABLE.ExceptionPatientTest.DivisionCode);

			this.ExceptionData = dtExceptionUnitTest.NewRow();
			this.ExceptionData[TABLE.ExceptionPatientTest.ExceptionPatientTestGuid] = Guid.NewGuid();
			this.ExceptionData[TABLE.ExceptionPatientTest.ExceptionTypeId] = this.ExceptionType;
			this.ExceptionData[TABLE.ExceptionPatientTest.PatientGuid] = DBNull.Value;
			this.ExceptionData[TABLE.ExceptionPatientTest.PatientName] = "";
			this.ExceptionData[TABLE.ExceptionPatientTest.PatientId] = "";
			this.ExceptionData[TABLE.ExceptionPatientTest.SpecimenUid] = "";
			this.ExceptionData[TABLE.ExceptionPatientTest.UnitGuid] = bloodUnitGuid;
			this.ExceptionData[TABLE.ExceptionPatientTest.UnitId] = bloodUnitId;
			this.ExceptionData[TABLE.ExceptionPatientTest.TestName] = testName;
			this.ExceptionData[TABLE.ExceptionPatientTest.ReactivityPhase] = phaseOfReaction;
			this.ExceptionData[TABLE.ExceptionPatientTest.TestInterp] = testInterp;
			this.ExceptionData[TABLE.ExceptionPatientTest.TestDate] = testDate;
			this.ExceptionData[TABLE.ExceptionPatientTest.TestingTechName] = GetUserNameFromID(testingTech);
			this.ExceptionData[TABLE.ExceptionPatientTest.ExceptionUser] = this.ExceptionUser;
			this.ExceptionData[TABLE.ExceptionPatientTest.ExceptionComment] = this.ExceptionComment;
			this.ExceptionData[TABLE.ExceptionPatientTest.ExceptionDate] = this.ExceptionDate;
			this.ExceptionData[TABLE.ExceptionPatientTest.DivisionCode] = Common.LogonUser.LogonUserDivisionCode;
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/8/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6913"> 
		///		<ExpectedInput>Valid patient and unit guids and test details</ExpectedInput>
		///		<ExpectedOutput>Exception report entry created</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="6914"> 
		///		<ExpectedInput>Invlaid testing tech name of null</ExpectedInput>
		///		<ExpectedOutput>Null Argument Exception thrown</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Generates exception for problems in testing for emrgency issued units 
		/// Implements BR_56.29
		/// </summary>
		/// <param name="bloodUnitGuid">Blood Unit Guid</param>
		/// <param name="patientGuid">Patient Guid</param>
		/// <param name="testName">Test Name</param>
		/// <param name="testInterp">Test Interp</param>
		/// <param name="testingTech">Testing Tech</param>
		public void GenerateExceptionEmergencyIssueUnitTest(Guid bloodUnitGuid, Guid patientGuid, string testName, string testInterp, string testingTech)
		{
			BOL.Patient patient = new BOL.Patient(patientGuid);
			BOL.BloodUnit bloodUnit = new BOL.BloodUnit(bloodUnitGuid);

			DataTable dtExceptionUnitTest = new DataTable(TABLE.ExceptionEmergencyIssueUnitTest.TableName);
			dtExceptionUnitTest.Columns.Add(TABLE.ExceptionEmergencyIssueUnitTest.ExceptionEmergencyIssueUnitTestGuid, typeof(Guid));
			dtExceptionUnitTest.Columns.Add(TABLE.ExceptionEmergencyIssueUnitTest.ExceptionTypeId, typeof(int));
			dtExceptionUnitTest.Columns.Add(TABLE.ExceptionEmergencyIssueUnitTest.PatientGuid, typeof(Guid));
			dtExceptionUnitTest.Columns.Add(TABLE.ExceptionEmergencyIssueUnitTest.PatientName);
			dtExceptionUnitTest.Columns.Add(TABLE.ExceptionEmergencyIssueUnitTest.PatientId);
			dtExceptionUnitTest.Columns.Add(TABLE.ExceptionEmergencyIssueUnitTest.PatientAboRh);
			dtExceptionUnitTest.Columns.Add(TABLE.ExceptionEmergencyIssueUnitTest.TestPerformed);
			dtExceptionUnitTest.Columns.Add(TABLE.ExceptionEmergencyIssueUnitTest.TestInterp);
			dtExceptionUnitTest.Columns.Add(TABLE.ExceptionEmergencyIssueUnitTest.UnitGuid, typeof(Guid));
			dtExceptionUnitTest.Columns.Add(TABLE.ExceptionEmergencyIssueUnitTest.UnitId);
			dtExceptionUnitTest.Columns.Add(TABLE.ExceptionEmergencyIssueUnitTest.UnitProductType);
			dtExceptionUnitTest.Columns.Add(TABLE.ExceptionEmergencyIssueUnitTest.UnitAboRh);
			dtExceptionUnitTest.Columns.Add(TABLE.ExceptionEmergencyIssueUnitTest.UnitExpirationDate, typeof(DateTime));
			dtExceptionUnitTest.Columns.Add(TABLE.ExceptionEmergencyIssueUnitTest.TestingTechName);
			dtExceptionUnitTest.Columns.Add(TABLE.ExceptionEmergencyIssueUnitTest.ExceptionUser);
			dtExceptionUnitTest.Columns.Add(TABLE.ExceptionEmergencyIssueUnitTest.ExceptionComment);
			dtExceptionUnitTest.Columns.Add(TABLE.ExceptionEmergencyIssueUnitTest.ExceptionDate, typeof(DateTime));
			dtExceptionUnitTest.Columns.Add(TABLE.ExceptionEmergencyIssueUnitTest.DivisionCode);

			this.ExceptionData = dtExceptionUnitTest.NewRow();
			this.ExceptionData[TABLE.ExceptionEmergencyIssueUnitTest.ExceptionEmergencyIssueUnitTestGuid] = Guid.NewGuid();
			this.ExceptionData[TABLE.ExceptionEmergencyIssueUnitTest.ExceptionTypeId] = this.ExceptionType;
			this.ExceptionData[TABLE.ExceptionEmergencyIssueUnitTest.PatientGuid] = patientGuid;
			this.ExceptionData[TABLE.ExceptionEmergencyIssueUnitTest.PatientName] = patient.LastName + ", " + patient.FirstName + " " + patient.MiddleInitial;
			this.ExceptionData[TABLE.ExceptionEmergencyIssueUnitTest.PatientId] = patient.SSN;
			this.ExceptionData[TABLE.ExceptionEmergencyIssueUnitTest.PatientAboRh] = patient.AboRh.AboRHText;
			this.ExceptionData[TABLE.ExceptionEmergencyIssueUnitTest.TestPerformed] = testName;
			this.ExceptionData[TABLE.ExceptionEmergencyIssueUnitTest.TestInterp] = testInterp;
			this.ExceptionData[TABLE.ExceptionEmergencyIssueUnitTest.UnitGuid] = bloodUnitGuid;
			this.ExceptionData[TABLE.ExceptionEmergencyIssueUnitTest.UnitId] = bloodUnit.EyeReadableUnitId;
			this.ExceptionData[TABLE.ExceptionEmergencyIssueUnitTest.UnitProductType] = bloodUnit.ProductType.ProductTypeText;
			this.ExceptionData[TABLE.ExceptionEmergencyIssueUnitTest.UnitAboRh] = bloodUnit.BloodUnitMedia.AboRh.AboRHText;
			this.ExceptionData[TABLE.ExceptionEmergencyIssueUnitTest.UnitExpirationDate] = bloodUnit.BloodUnitMedia.UnitExpirationDate;
			this.ExceptionData[TABLE.ExceptionEmergencyIssueUnitTest.TestingTechName] = GetUserNameFromID(testingTech);
			this.ExceptionData[TABLE.ExceptionEmergencyIssueUnitTest.ExceptionUser] = this.ExceptionUser;
			this.ExceptionData[TABLE.ExceptionEmergencyIssueUnitTest.ExceptionComment] = this.ExceptionComment;
			this.ExceptionData[TABLE.ExceptionEmergencyIssueUnitTest.ExceptionDate] = this.ExceptionDate;
			this.ExceptionData[TABLE.ExceptionEmergencyIssueUnitTest.DivisionCode] = Common.LogonUser.LogonUserDivisionCode;
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/9/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="7469"> 
		///		<ExpectedInput>Valid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="7470"> 
		///		<ExpectedInput>Invalid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Generate exception when expired order is selected
		/// BR_92.09
		/// </summary>
		/// <param name="orderedTest"></param>
		/// <param name="orderedComponent"></param>
		/// <param name="testingTech"></param>
		public void GenerateExceptionExpiredTask(BOL.OrderedTest orderedTest, BOL.OrderedComponent orderedComponent, string testingTech)
		{
			DataTable dtExceptionUnitTest = new DataTable(TABLE.ExceptionExpiredTask.TableName);
			dtExceptionUnitTest.Columns.Add(TABLE.ExceptionExpiredTask.ExceptionExpiredTaskGuid, typeof(Guid));
			dtExceptionUnitTest.Columns.Add(TABLE.ExceptionExpiredTask.ExceptionTypeId, typeof(int));
			dtExceptionUnitTest.Columns.Add(TABLE.ExceptionExpiredTask.PatientName);
			dtExceptionUnitTest.Columns.Add(TABLE.ExceptionExpiredTask.PatientId);
			dtExceptionUnitTest.Columns.Add(TABLE.ExceptionExpiredTask.CprsOrderNumber);
			dtExceptionUnitTest.Columns.Add(TABLE.ExceptionExpiredTask.SpecimenUid);
			dtExceptionUnitTest.Columns.Add(TABLE.ExceptionExpiredTask.ComponentClassOrdered);
			dtExceptionUnitTest.Columns.Add(TABLE.ExceptionExpiredTask.Quantity);
			dtExceptionUnitTest.Columns.Add(TABLE.ExceptionExpiredTask.DateWanted, typeof(DateTime));
			dtExceptionUnitTest.Columns.Add(TABLE.ExceptionExpiredTask.LoginTechName);
			dtExceptionUnitTest.Columns.Add(TABLE.ExceptionExpiredTask.ExceptionUser);
			dtExceptionUnitTest.Columns.Add(TABLE.ExceptionExpiredTask.ExceptionComment);
			dtExceptionUnitTest.Columns.Add(TABLE.ExceptionExpiredTask.ExceptionDate, typeof(DateTime));
			dtExceptionUnitTest.Columns.Add(TABLE.ExceptionExpiredTask.DivisionCode);
			//
			this.ExceptionData = dtExceptionUnitTest.NewRow();
			this.ExceptionData[TABLE.ExceptionExpiredTask.ExceptionExpiredTaskGuid] = Guid.NewGuid();
			this.ExceptionData[TABLE.ExceptionExpiredTask.ExceptionTypeId] = this.ExceptionType;
			if (orderedTest != null)
			{
				BOL.Patient patient = new BOL.Patient(orderedTest.PatientGuid);
				this.ExceptionData[TABLE.ExceptionExpiredTask.PatientName] = patient.LastName + ", " + patient.FirstName + " " + patient.MiddleInitial;
				this.ExceptionData[TABLE.ExceptionExpiredTask.PatientId] = patient.SSN;
				this.ExceptionData[TABLE.ExceptionExpiredTask.CprsOrderNumber] = orderedTest.OrderGroupNumber;
				if (orderedTest.Specimen != null)
					this.ExceptionData[TABLE.ExceptionExpiredTask.SpecimenUid] = orderedTest.Specimen.SpecimenUid;
				else
					this.ExceptionData[TABLE.ExceptionExpiredTask.SpecimenUid] = "";
				this.ExceptionData[TABLE.ExceptionExpiredTask.ComponentClassOrdered] = "";
				this.ExceptionData[TABLE.ExceptionExpiredTask.Quantity] = 0;
				//
				// CR 3298
				if (orderedTest.RequiredResultsDate != null && orderedTest.RequiredResultsDate != DateTime.MinValue)
				{
					this.ExceptionData[TABLE.ExceptionExpiredTask.DateWanted] = orderedTest.RequiredResultsDate;
				}
			}
			else
			{
				BOL.Patient patient = new BOL.Patient(orderedComponent.PatientGuid);
				this.ExceptionData[TABLE.ExceptionExpiredTask.PatientName] = patient.LastName + ", " + patient.FirstName + " " + patient.MiddleInitial;
				this.ExceptionData[TABLE.ExceptionExpiredTask.PatientId] = patient.SSN;
				this.ExceptionData[TABLE.ExceptionExpiredTask.CprsOrderNumber] = orderedComponent.OrderGroupNumber;
				if (orderedComponent.SpecimenIdentified) // CR3114
					this.ExceptionData[TABLE.ExceptionExpiredTask.SpecimenUid] = orderedComponent.GetSpecimen().SpecimenUid;  // CR3114
				else
					this.ExceptionData[TABLE.ExceptionExpiredTask.SpecimenUid] = "";
				this.ExceptionData[TABLE.ExceptionExpiredTask.ComponentClassOrdered] = orderedComponent.ComponentOrdered.BloodComponentName;
				this.ExceptionData[TABLE.ExceptionExpiredTask.Quantity] = orderedComponent.Quantity;
				//
				// CR 3298
				if (orderedComponent.RequiredDateTime != null && orderedComponent.RequiredDateTime != DateTime.MinValue)
				{
					this.ExceptionData[TABLE.ExceptionExpiredTask.DateWanted] = orderedComponent.RequiredDateTime;
				}
			}
			//
			this.ExceptionData[TABLE.ExceptionExpiredTask.LoginTechName] = testingTech;
			this.ExceptionData[TABLE.ExceptionExpiredTask.ExceptionUser] = this.ExceptionUser;
			this.ExceptionData[TABLE.ExceptionExpiredTask.ExceptionComment] = this.ExceptionComment;
			this.ExceptionData[TABLE.ExceptionExpiredTask.ExceptionDate] = this.ExceptionDate;
			this.ExceptionData[TABLE.ExceptionExpiredTask.DivisionCode] = Common.LogonUser.LogonUserDivisionCode;
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/9/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="7471"> 
		///		<ExpectedInput>Valid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="7472"> 
		///		<ExpectedInput>Invalid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Generates inconclusive crossmatch exception
		/// BR_40.20
		/// </summary>
		/// <param name="patientOrder"></param>
		/// <param name="bloodUnitGuid"></param>
		/// <param name="systemInterp"></param>
		/// <param name="techInterp"></param>
		/// <param name="testingTech"></param>
		public void GenerateExceptionInconclusiveCrossmatch(BOL.PatientOrder patientOrder, Guid bloodUnitGuid, string systemInterp, string techInterp, string testingTech)
		{
			BOL.OrderedTest orderedTest = null;
			BOL.OrderedComponent orderedComponent = null;
			BOL.BloodUnit bloodUnit = new BOL.BloodUnit(bloodUnitGuid);

			if (patientOrder == null)
			{
				this.ExceptionData = null;
				return;
			}
			else
			{
				if (patientOrder is BOL.OrderedTest)
				{
					orderedTest = (BOL.OrderedTest)patientOrder;
				}
				else
				{
					orderedComponent = (BOL.OrderedComponent)patientOrder;
				}
			}

			DataTable dtExceptionInconclusiveCrossmatch = new DataTable(TABLE.ExceptionInconclusiveCrossmatch.TableName);
			dtExceptionInconclusiveCrossmatch.Columns.Add(TABLE.ExceptionInconclusiveCrossmatch.ExceptionInconclusiveCrossmatchGuid, typeof(Guid));
			dtExceptionInconclusiveCrossmatch.Columns.Add(TABLE.ExceptionInconclusiveCrossmatch.ExceptionTypeId, typeof(int));
			dtExceptionInconclusiveCrossmatch.Columns.Add(TABLE.ExceptionInconclusiveCrossmatch.PatientGuid, typeof(Guid));
			dtExceptionInconclusiveCrossmatch.Columns.Add(TABLE.ExceptionInconclusiveCrossmatch.PatientName);
			dtExceptionInconclusiveCrossmatch.Columns.Add(TABLE.ExceptionInconclusiveCrossmatch.PatientId);
			dtExceptionInconclusiveCrossmatch.Columns.Add(TABLE.ExceptionInconclusiveCrossmatch.SpecimenUid);
			dtExceptionInconclusiveCrossmatch.Columns.Add(TABLE.ExceptionInconclusiveCrossmatch.BloodUnitGuid, typeof(Guid));
			dtExceptionInconclusiveCrossmatch.Columns.Add(TABLE.ExceptionInconclusiveCrossmatch.BloodUnitId);
			dtExceptionInconclusiveCrossmatch.Columns.Add(TABLE.ExceptionInconclusiveCrossmatch.UnitProductCode);
			dtExceptionInconclusiveCrossmatch.Columns.Add(TABLE.ExceptionInconclusiveCrossmatch.SystemInterp);
			dtExceptionInconclusiveCrossmatch.Columns.Add(TABLE.ExceptionInconclusiveCrossmatch.TechInterp);
			dtExceptionInconclusiveCrossmatch.Columns.Add(TABLE.ExceptionInconclusiveCrossmatch.TestingTechName);
			dtExceptionInconclusiveCrossmatch.Columns.Add(TABLE.ExceptionInconclusiveCrossmatch.ExceptionUser);
			dtExceptionInconclusiveCrossmatch.Columns.Add(TABLE.ExceptionInconclusiveCrossmatch.ExceptionComment);
			dtExceptionInconclusiveCrossmatch.Columns.Add(TABLE.ExceptionInconclusiveCrossmatch.ExceptionDate, typeof(DateTime));
			dtExceptionInconclusiveCrossmatch.Columns.Add(TABLE.ExceptionInconclusiveCrossmatch.DivisionCode);

			this.ExceptionData = dtExceptionInconclusiveCrossmatch.NewRow();
			this.ExceptionData[TABLE.ExceptionInconclusiveCrossmatch.ExceptionInconclusiveCrossmatchGuid] = Guid.NewGuid();
			this.ExceptionData[TABLE.ExceptionInconclusiveCrossmatch.ExceptionTypeId] = this.ExceptionType;
			
			if (orderedTest != null)
			{
				this.ExceptionData[TABLE.ExceptionInconclusiveCrossmatch.PatientGuid] = orderedTest.Patient.PatientGuid;
				this.ExceptionData[TABLE.ExceptionInconclusiveCrossmatch.PatientName] = orderedTest.Patient.DisplayName;
				this.ExceptionData[TABLE.ExceptionInconclusiveCrossmatch.PatientId] = orderedTest.Patient.DisplayPatientID;
				this.ExceptionData[TABLE.ExceptionInconclusiveCrossmatch.SpecimenUid] = orderedTest.Specimen.SpecimenUid;
			}
            else if (orderedComponent != null) //CR 3554
			{
				this.ExceptionData[TABLE.ExceptionInconclusiveCrossmatch.PatientGuid] = orderedComponent.Patient.PatientGuid;
				this.ExceptionData[TABLE.ExceptionInconclusiveCrossmatch.PatientName] = orderedComponent.Patient.DisplayName;
				this.ExceptionData[TABLE.ExceptionInconclusiveCrossmatch.PatientId] = orderedComponent.Patient.DisplayPatientID;
				this.ExceptionData[TABLE.ExceptionInconclusiveCrossmatch.SpecimenUid] = orderedComponent.GetSpecimen().SpecimenUid;  // CR3114
			}

			this.ExceptionData[TABLE.ExceptionInconclusiveCrossmatch.BloodUnitId] = bloodUnit.EyeReadableUnitId;
			this.ExceptionData[TABLE.ExceptionInconclusiveCrossmatch.UnitProductCode] = bloodUnit.ProductCode;
			this.ExceptionData[TABLE.ExceptionInconclusiveCrossmatch.BloodUnitGuid] = bloodUnitGuid;

			this.ExceptionData[TABLE.ExceptionInconclusiveCrossmatch.SystemInterp] = systemInterp;
			this.ExceptionData[TABLE.ExceptionInconclusiveCrossmatch.TechInterp] = techInterp;
			this.ExceptionData[TABLE.ExceptionInconclusiveCrossmatch.TestingTechName] = GetUserNameFromID(testingTech);
			this.ExceptionData[TABLE.ExceptionInconclusiveCrossmatch.ExceptionUser] = this.ExceptionUser;
			this.ExceptionData[TABLE.ExceptionInconclusiveCrossmatch.ExceptionComment] = this.ExceptionComment;
			this.ExceptionData[TABLE.ExceptionInconclusiveCrossmatch.ExceptionDate] = this.ExceptionDate;
			this.ExceptionData[TABLE.ExceptionInconclusiveCrossmatch.DivisionCode] = Common.LogonUser.LogonUserDivisionCode;
		}

		/// <summary>
		/// Gets user name from user id
		/// </summary>
		/// <param name="userID"></param>
		private string GetUserNameFromID(string userID)
		{
			string retValue = userID;

			//04/23/2005 - Lohse trying to fix this so it can handle names or IDs -- 
			//Which PERSONALLY, I feel is WRONG!  But, it's alot of work to type up a email to the Enginerds
			System.Collections.ArrayList alUsers = BOL.VbecsUser.GetActiveDivisionUsers(Common.LogonUser.LogonUserDivisionCode);

			foreach(BOL.VbecsUser tmpUser in alUsers)
			{
				if (tmpUser.UserName == userID)
					break;
				
				if (tmpUser.UserNTLoginID != null)
				{
					if (tmpUser.UserNTLoginID.ToUpper().Trim() == userID.ToUpper().Trim())
					{
						retValue = tmpUser.UserName;
						break;
					}
				}
			}

			return(retValue);
		}

		/// <summary>
		/// Combines Abo and Rh into one string
		/// </summary>
		/// <param name="Abo"></param>
		/// <param name="Rh"></param>
		/// <returns></returns>
		private string CombineAboRh(string Abo, string Rh)
		{
			string AboRh = "";

			if (Rh.Trim() == "P")
				AboRh = Abo.Trim() + " Pos";
			else if (Rh.Trim() == "N")
				AboRh = Abo.Trim() + " Neg";
			else if (Rh.Trim() == "I")
				AboRh = Abo.Trim() + " Inc";
			else
				AboRh = Abo.Trim() + " " + Rh.Trim();

			return AboRh;
		}

		///<Developers>
		///	<Developer>Krzysztof Dobranowski</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/7/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="2292"> 
		///		<ExpectedInput>Valid startDate, endDate and ExceptionType</ExpectedInput>
		///		<ExpectedOutput>DataSet</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="2296"> 
		///		<ExpectedInput>Invalid ExceptionReportType</ExpectedInput>
		///		<ExpectedOutput>Empty DataSet</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Retrieve and prepare exception data to be used as a source for SSRS report
		/// </summary>
		/// <param name="startDate"></param>
		/// <param name="endDate"></param>
		/// <param name="exceptionType"></param>
		/// <returns></returns>
		public static DataSet GetExceptionReport(DateTime startDate, DateTime endDate, Common.ExceptionReportType exceptionType)
		{
			DataSet dsExceptionReport = null;
			switch(exceptionType)
			{
				case Common.ExceptionReportType.ExceptionReportLabelMismatch:
					dsExceptionReport = DAL.ExceptionReport.GetExceptionReportLabelMismatch(Common.LogonUser.LogonUserDivisionCode, startDate, endDate);
					break;
				case Common.ExceptionReportType.ExceptionReportPatientAboRh:
					dsExceptionReport = DAL.ExceptionReport.GetExceptionReportPatientAboRH(Common.LogonUser.LogonUserDivisionCode, startDate, endDate);
					break;
				case Common.ExceptionReportType.ExceptionReportPatientAboRhDiscrepancy:
					dsExceptionReport = DAL.ExceptionReport.GetExceptionReportPatientAboRHDiscrepancy(Common.LogonUser.LogonUserDivisionCode, startDate, endDate);
					break;
				case Common.ExceptionReportType.ExceptionReportPatientAntigenDiscrepancy:
					dsExceptionReport = DAL.ExceptionReport.GetExceptionReportPatientAntigenDiscrepancy(Common.LogonUser.LogonUserDivisionCode, startDate, endDate);
					break;
				case Common.ExceptionReportType.ExceptionReportPatientTest:
					dsExceptionReport = DAL.ExceptionReport.GetExceptionReportPatientTest(Common.LogonUser.LogonUserDivisionCode, startDate, endDate);
					break;
				case Common.ExceptionReportType.ExceptionReportPatientTestInactivated:
					dsExceptionReport = DAL.ExceptionReport.GetExceptionReportPatientTestInactivated(Common.LogonUser.LogonUserDivisionCode, startDate, endDate);
					break;
				case Common.ExceptionReportType.ExceptionReportQC:
					dsExceptionReport = DAL.ExceptionReport.GetExceptionReportQC(Common.LogonUser.LogonUserDivisionCode, startDate, endDate);
					break;
				case Common.ExceptionReportType.ExceptionReportQCInactivated:
					dsExceptionReport = DAL.ExceptionReport.GetExceptionReportQCInactivated(Common.LogonUser.LogonUserDivisionCode, startDate, endDate);
					break;
				case Common.ExceptionReportType.ExceptionReportQCNotPerformed:
					dsExceptionReport = DAL.ExceptionReport.GetExceptionReportQCNotPerformed(Common.LogonUser.LogonUserDivisionCode, startDate, endDate);
					break;
				case Common.ExceptionReportType.ExceptionReportQuarantine:
					dsExceptionReport = DAL.ExceptionReport.GetExceptionReportQuarantine(Common.LogonUser.LogonUserDivisionCode, startDate, endDate);
					break;
				case Common.ExceptionReportType.ExceptionReportReflexTest:
					dsExceptionReport = DAL.ExceptionReport.GetExceptionReportReflexTest(Common.LogonUser.LogonUserDivisionCode, startDate, endDate);
					break;
				case Common.ExceptionReportType.ExceptionReportSpecimen:
					dsExceptionReport = DAL.ExceptionReport.GetExceptionReportSpecimen(Common.LogonUser.LogonUserDivisionCode, startDate, endDate);
					break;
				case Common.ExceptionReportType.ExceptionReportUnitAboRh:
					dsExceptionReport = DAL.ExceptionReport.GetExceptionReportUnitAboRH(Common.LogonUser.LogonUserDivisionCode, startDate, endDate);
					break;
				case Common.ExceptionReportType.ExceptionReportUnitModification:
					dsExceptionReport = DAL.ExceptionReport.GetExceptionReportUnitModification(Common.LogonUser.LogonUserDivisionCode, startDate, endDate);
					break;
				case Common.ExceptionReportType.ExceptionReportUnitSelection:
					dsExceptionReport = DAL.ExceptionReport.GetExceptionReportUnitSelection(Common.LogonUser.LogonUserDivisionCode, startDate, endDate);
					break;
				case Common.ExceptionReportType.ExceptionReportReturnIssuedUnit:
					dsExceptionReport = DAL.ExceptionReport.GetExceptionReportReturnIssuedUnit(Common.LogonUser.LogonUserDivisionCode, startDate, endDate);
					break;
				case Common.ExceptionReportType.ExceptionReportAboIncompatibleTransfusion:
					dsExceptionReport = DAL.ExceptionReport.GetExceptionReportAboIncompatibleTransfusion(Common.LogonUser.LogonUserDivisionCode, startDate, endDate);
					break;
				case Common.ExceptionReportType.ExceptionReportExpiredUnitReceived:
					dsExceptionReport = DAL.ExceptionReport.GetExceptionReportExpiredUnitReceived(Common.LogonUser.LogonUserDivisionCode, startDate, endDate);
					break;
				case Common.ExceptionReportType.ExceptionReportCautionTagBTRFPrinting:
					dsExceptionReport = DAL.ExceptionReport.GetExceptionReportCautionTagBTRF(Common.LogonUser.LogonUserDivisionCode, startDate, endDate);
					break;
				case Common.ExceptionReportType.ExceptionReportTrwSpecimen:
					dsExceptionReport = DAL.ExceptionReport.GetExceptionReportTrwSpecimen(Common.LogonUser.LogonUserDivisionCode, startDate, endDate);
					break;
				case Common.ExceptionReportType.ExceptionReportTransfusionReactionWorkup:
					dsExceptionReport = DAL.ExceptionReport.GetExceptionReportTransfusionReactionWorkup(Common.LogonUser.LogonUserDivisionCode, startDate, endDate);
					break;
				case Common.ExceptionReportType.ExceptionReportAntigenTest:
					dsExceptionReport = DAL.ExceptionReport.GetExceptionReportAntigenTestInvalidated(Common.LogonUser.LogonUserDivisionCode, startDate, endDate);
					break;
				case Common.ExceptionReportType.ExceptionReportUnitTest:
					dsExceptionReport = DAL.ExceptionReport.GetExceptionReportUnitTest(Common.LogonUser.LogonUserDivisionCode, startDate, endDate);
					break;
				case Common.ExceptionReportType.ExceptionReportTransfusion:
					dsExceptionReport = DAL.ExceptionReport.GetExceptionReportTransfusion(Common.LogonUser.LogonUserDivisionCode, startDate, endDate);
					break;
				case Common.ExceptionReportType.ExceptionReportExpiredTaskProcessed:
					dsExceptionReport = DAL.ExceptionReport.GetExceptionReportExpiredTask(Common.LogonUser.LogonUserDivisionCode, startDate, endDate);
					break;
				case Common.ExceptionReportType.ExceptionReportUnitERIssuedTestingProblem:
					dsExceptionReport = DAL.ExceptionReport.GetExceptionReportEmergencyIssueUnitTest(Common.LogonUser.LogonUserDivisionCode, startDate, endDate);
					break;
				case Common.ExceptionReportType.ExceptionReportUnitSupply:
					dsExceptionReport = DAL.ExceptionReport.GetExceptionReportUnitSupply(Common.LogonUser.LogonUserDivisionCode, startDate, endDate);
					break;
				case Common.ExceptionReportType.ExceptionReportUnitIssue:
					dsExceptionReport = DAL.ExceptionReport.GetExceptionReportUnitIssue(Common.LogonUser.LogonUserDivisionCode, startDate, endDate);
					break;
				case Common.ExceptionReportType.ExceptionReportEmergencyIssueOrder:
					dsExceptionReport = DAL.ExceptionReport.GetExceptionReportEmergencyIssueOrder(Common.LogonUser.LogonUserDivisionCode, startDate, endDate);
					break;
				case Common.ExceptionReportType.ExceptionReportPatientRestriction:
					dsExceptionReport = DAL.ExceptionReport.GetExceptionReportPatientRestriction(Common.LogonUser.LogonUserDivisionCode, startDate, endDate);
					break;
				case Common.ExceptionReportType.ExceptionReportRemoveFinalUnitStatus:
					dsExceptionReport = DAL.ExceptionReport.GetExceptionReportRemoveFinalUnitStatus(Common.LogonUser.LogonUserDivisionCode, startDate, endDate);
					break;
				case Common.ExceptionReportType.ExceptionReportCrossmatch:
					dsExceptionReport = DAL.ExceptionReport.GetExceptionReportInconclusiveCrossmatch(Common.LogonUser.LogonUserDivisionCode, startDate, endDate);
					break;
				//CR 2942
				case Common.ExceptionReportType.ExceptionReportElectronicTransfusion:
					dsExceptionReport = DAL.ExceptionReport.GetExceptionReportElectronicTransfusion(Common.LogonUser.LogonUserDivisionCode, startDate, endDate);
					break;
				default:
					dsExceptionReport = new DataSet();
					break;
			}
			return dsExceptionReport;
		}

		///<Developers>
		///	<Developer>Krzysztof Dobranowski</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/7/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="2297"> 
		///		<ExpectedInput>Valid startDate, endDate and ExceptionType</ExpectedInput>
		///		<ExpectedOutput>DataSet</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="2306"> 
		///		<ExpectedInput>Invalid ExceptionReportType</ExpectedInput>
		///		<ExpectedOutput>Empty DataSet</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Retrieve and prepare exception data for a single patient to be used as a source for SSRS report 
		/// </summary>
		/// <param name="patientGuid">Patient unique identifier</param>
		/// <param name="exceptionType">Exception Type</param>
		/// <returns></returns>
		public static DataSet GetPatientExceptionReport(Guid patientGuid, Common.ExceptionReportType exceptionType)
		{
			DataSet dsExceptionReport = null;
			switch(exceptionType)
			{
				case Common.ExceptionReportType.ExceptionReportPatientAboRh:
					dsExceptionReport = DAL.ExceptionReport.GetExceptionReportPatientAboRH(Common.LogonUser.LogonUserDivisionCode, patientGuid);
					break;
				case Common.ExceptionReportType.ExceptionReportPatientAboRhDiscrepancy:
					dsExceptionReport = DAL.ExceptionReport.GetExceptionReportPatientAboRHDiscrepancy(Common.LogonUser.LogonUserDivisionCode, patientGuid);
					break;
				case Common.ExceptionReportType.ExceptionReportPatientAntigenDiscrepancy:
					dsExceptionReport = DAL.ExceptionReport.GetExceptionReportPatientAntigenDiscrepancy(Common.LogonUser.LogonUserDivisionCode, patientGuid);
					break;
				case Common.ExceptionReportType.ExceptionReportPatientTest:
					dsExceptionReport = DAL.ExceptionReport.GetExceptionReportPatientTest(Common.LogonUser.LogonUserDivisionCode, patientGuid);
					break;
				case Common.ExceptionReportType.ExceptionReportPatientTestInactivated:
					dsExceptionReport = DAL.ExceptionReport.GetExceptionReportPatientTestInactivated(Common.LogonUser.LogonUserDivisionCode, patientGuid);
					break;
				case Common.ExceptionReportType.ExceptionReportQCNotPerformed:
					dsExceptionReport = DAL.ExceptionReport.GetExceptionReportQCNotPerformed(Common.LogonUser.LogonUserDivisionCode, patientGuid);
					break;
				case Common.ExceptionReportType.ExceptionReportQC:
					dsExceptionReport = DAL.ExceptionReport.GetExceptionReportQC(Common.LogonUser.LogonUserDivisionCode, patientGuid);
					break;
				case Common.ExceptionReportType.ExceptionReportReflexTest:
					dsExceptionReport = DAL.ExceptionReport.GetExceptionReportReflexTest(Common.LogonUser.LogonUserDivisionCode, patientGuid);
					break;
				case Common.ExceptionReportType.ExceptionReportSpecimen:
					dsExceptionReport = DAL.ExceptionReport.GetExceptionReportSpecimen(Common.LogonUser.LogonUserDivisionCode, patientGuid);
					break;
				case Common.ExceptionReportType.ExceptionReportUnitSelection:
					dsExceptionReport = DAL.ExceptionReport.GetExceptionReportUnitSelection(Common.LogonUser.LogonUserDivisionCode, patientGuid);
					break;
				case Common.ExceptionReportType.ExceptionReportReturnIssuedUnit:
					dsExceptionReport = DAL.ExceptionReport.GetExceptionReportReturnIssuedUnit(Common.LogonUser.LogonUserDivisionCode, patientGuid);
					break;
				case Common.ExceptionReportType.ExceptionReportAboIncompatibleTransfusion:
					dsExceptionReport = DAL.ExceptionReport.GetExceptionReportAboIncompatibleTransfusion(Common.LogonUser.LogonUserDivisionCode, patientGuid);
					break;
				case Common.ExceptionReportType.ExceptionReportTrwSpecimen:
					dsExceptionReport = DAL.ExceptionReport.GetExceptionReportTrwSpecimen(Common.LogonUser.LogonUserDivisionCode, patientGuid);
					break;
				case Common.ExceptionReportType.ExceptionReportTransfusionReactionWorkup:
					dsExceptionReport = DAL.ExceptionReport.GetExceptionReportTransfusionReactionWorkup(Common.LogonUser.LogonUserDivisionCode, patientGuid);
					break;
				case Common.ExceptionReportType.ExceptionReportCautionTagBTRFPrinting:
					dsExceptionReport = DAL.ExceptionReport.GetExceptionReportCautionTagBTRF(Common.LogonUser.LogonUserDivisionCode, patientGuid);
					break;
				case Common.ExceptionReportType.ExceptionReportTransfusion:
					dsExceptionReport = DAL.ExceptionReport.GetExceptionReportTransfusion(Common.LogonUser.LogonUserDivisionCode, patientGuid);
					break;
				case Common.ExceptionReportType.ExceptionReportPatientRestriction:
					dsExceptionReport = DAL.ExceptionReport.GetExceptionReportPatientRestriction(Common.LogonUser.LogonUserDivisionCode, patientGuid);
					break;
				default:
					dsExceptionReport = new DataSet();
					break;
			}
			return dsExceptionReport;
		}

		///<Developers>
		///	<Developer>David Askew</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>10/7/2004</CreationDate>
		///<TestCases>
		///
		///<Case type="0" testid ="5765"> 
		///		<ExpectedInput>Valid startDate, endDate and ExceptionType</ExpectedInput>
		///		<ExpectedOutput>DataSet</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5766"> 
		///		<ExpectedInput>Invalid ExceptionType</ExpectedInput>
		///		<ExpectedOutput>Empty DataSet</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Retrieve exception data for Blood Unit History report 
		/// </summary>
		/// <param name="bloodUnitGuid">Blood Unit unique id</param>
		/// <param name="exceptionType">Exception Type</param>
		/// <returns></returns>
		public static DataSet GetBloodUnitExceptionReport(Guid bloodUnitGuid, Common.ExceptionReportType exceptionType)
		{
			DataSet dsExceptionReport = null;
			switch(exceptionType)
			{
				case Common.ExceptionReportType.ExceptionReportAntigenTest:
					dsExceptionReport = DAL.ExceptionReport.GetExceptionReportAntigenTestInvalidated(bloodUnitGuid);
					break;
				case Common.ExceptionReportType.ExceptionReportExpiredUnitReceived:
					dsExceptionReport = DAL.ExceptionReport.GetExceptionReportExpiredUnitReceived(bloodUnitGuid);
					break;
				case Common.ExceptionReportType.ExceptionReportLabelMismatch:
					dsExceptionReport = DAL.ExceptionReport.GetExceptionReportLabelMismatch(bloodUnitGuid);
					break;
				case Common.ExceptionReportType.ExceptionReportQuarantine:
					dsExceptionReport = DAL.ExceptionReport.GetExceptionReportQuarantine(bloodUnitGuid);
					break;
				case Common.ExceptionReportType.ExceptionReportRemoveFinalUnitStatus:
					dsExceptionReport = DAL.ExceptionReport.GetExceptionReportRemoveFinalUnitStatus(bloodUnitGuid);
					break;
				case Common.ExceptionReportType.ExceptionReportReturnIssuedUnit:
					dsExceptionReport = DAL.ExceptionReport.GetExceptionReportReturnIssuedUnit(bloodUnitGuid);
					break;
				case Common.ExceptionReportType.ExceptionReportUnitAboRh:
					dsExceptionReport = DAL.ExceptionReport.GetExceptionReportUnitAboRH(bloodUnitGuid);
					break; 
				case Common.ExceptionReportType.ExceptionReportUnitModification:
					dsExceptionReport = DAL.ExceptionReport.GetExceptionReportUnitModification(bloodUnitGuid);
					break;
				case Common.ExceptionReportType.ExceptionReportUnitSelection:
					dsExceptionReport = DAL.ExceptionReport.GetExceptionReportUnitSelection(bloodUnitGuid);
					break;
				case Common.ExceptionReportType.ExceptionReportUnitSupply:
					dsExceptionReport = DAL.ExceptionReport.GetExceptionReportUnitSupply(bloodUnitGuid);
					break;
				case Common.ExceptionReportType.ExceptionReportQC:
					dsExceptionReport = DAL.ExceptionReport.GetExceptionReportQC(bloodUnitGuid);
					break;
				default:
					dsExceptionReport = new DataSet();
					break;
			}
			return dsExceptionReport;
		}

		///<Developers>
		///	<Developer>David Askew</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>01/27/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6622"> 
		///		<ExpectedInput>Valid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="6623"> 
		///		<ExpectedInput>Invalid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Generates exception for patient restriction removal
		/// Implements BR_71.04
		/// </summary>
		/// <param name="bloodUnit"></param>
		public void GenerateExceptionPatientRestriction(BloodUnit bloodUnit)
		{
			// Special Testing = CMV Neg + Sickle Cell Neg
			string specialTesting = string.Concat("CMV Neg: ", bloodUnit.CmvNegative ? "Yes," : "No,", " Sickle Cell Neg: ", bloodUnit.SickleCellNegative ? "Yes" : "No");

			DataTable dtExceptionPatientRestriction = new DataTable(TABLE.ExceptionPatientRestriction.TableName);
			dtExceptionPatientRestriction.Columns.Add(TABLE.ExceptionPatientRestriction.ExceptionPatientRestrictionGuid, typeof(System.Guid));
			dtExceptionPatientRestriction.Columns.Add(TABLE.ExceptionPatientRestriction.ExceptionTypeId, typeof(int));
			dtExceptionPatientRestriction.Columns.Add(TABLE.ExceptionPatientRestriction.UnitId, typeof(string));
			dtExceptionPatientRestriction.Columns.Add(TABLE.ExceptionPatientRestriction.UnitGuid, typeof(Guid));
			dtExceptionPatientRestriction.Columns.Add(TABLE.ExceptionPatientRestriction.UnitAboRh, typeof(string));
			dtExceptionPatientRestriction.Columns.Add(TABLE.ExceptionPatientRestriction.ProductShortName, typeof(string));
			dtExceptionPatientRestriction.Columns.Add(TABLE.ExceptionPatientRestriction.ProductCode, typeof(string));
			dtExceptionPatientRestriction.Columns.Add(TABLE.ExceptionPatientRestriction.UnitExpirationDate, typeof(DateTime));
			dtExceptionPatientRestriction.Columns.Add(TABLE.ExceptionPatientRestriction.SpecialTesting, typeof(string));
			dtExceptionPatientRestriction.Columns.Add(TABLE.ExceptionPatientRestriction.DonationType, typeof(string));
			dtExceptionPatientRestriction.Columns.Add(TABLE.ExceptionPatientRestriction.RestrictedPatientName, typeof(string));
			dtExceptionPatientRestriction.Columns.Add(TABLE.ExceptionPatientRestriction.RestrictedPatientId, typeof(string));
			dtExceptionPatientRestriction.Columns.Add(TABLE.ExceptionPatientRestriction.RestrictedPatientGuid, typeof(Guid));
			dtExceptionPatientRestriction.Columns.Add(TABLE.ExceptionPatientRestriction.ExceptionUser, typeof(string));
			dtExceptionPatientRestriction.Columns.Add(TABLE.ExceptionPatientRestriction.ExceptionComment, typeof(string));
			dtExceptionPatientRestriction.Columns.Add(TABLE.ExceptionPatientRestriction.ExceptionDate, typeof(DateTime));
			dtExceptionPatientRestriction.Columns.Add(TABLE.ExceptionPatientRestriction.DivisionCode, typeof(string)); 

			this.ExceptionData = dtExceptionPatientRestriction.NewRow();
			this.ExceptionData[TABLE.ExceptionPatientRestriction.ExceptionPatientRestrictionGuid] = System.Guid.NewGuid();
			this.ExceptionData[TABLE.ExceptionPatientRestriction.ExceptionTypeId] = this.ExceptionType;
			this.ExceptionData[TABLE.ExceptionPatientRestriction.UnitId] = bloodUnit.EyeReadableUnitId;
			this.ExceptionData[TABLE.ExceptionPatientRestriction.UnitGuid] = bloodUnit.BloodProductGuid;
			this.ExceptionData[TABLE.ExceptionPatientRestriction.UnitAboRh] = bloodUnit.BloodUnitMedia.AboRh.AboRHText;
			this.ExceptionData[TABLE.ExceptionPatientRestriction.ProductShortName] = bloodUnit.ProductShortName;
			this.ExceptionData[TABLE.ExceptionPatientRestriction.ProductCode] = bloodUnit.ProductCode;
			this.ExceptionData[TABLE.ExceptionPatientRestriction.UnitExpirationDate] = bloodUnit.BloodUnitMedia.UnitExpirationDate;
			this.ExceptionData[TABLE.ExceptionPatientRestriction.SpecialTesting] = specialTesting;
			this.ExceptionData[TABLE.ExceptionPatientRestriction.DonationType] = bloodUnit.DonationType.DonationTypeText;
			this.ExceptionData[TABLE.ExceptionPatientRestriction.RestrictedPatientName] = bloodUnit.DirectedPatient.DisplayName;
			this.ExceptionData[TABLE.ExceptionPatientRestriction.RestrictedPatientId] = bloodUnit.DirectedPatient.DisplayPatientID;
			this.ExceptionData[TABLE.ExceptionPatientRestriction.RestrictedPatientGuid] = bloodUnit.DirectedPatient.PatientGuid;
			this.ExceptionData[TABLE.ExceptionPatientRestriction.ExceptionUser] = this.ExceptionUser;
			this.ExceptionData[TABLE.ExceptionPatientRestriction.ExceptionComment] = this.ExceptionComment;
			this.ExceptionData[TABLE.ExceptionPatientRestriction.ExceptionDate] = this.ExceptionDate;
			this.ExceptionData[TABLE.ExceptionPatientRestriction.DivisionCode] = Common.LogonUser.LogonUserDivisionCode;
		}

		/// <summary>
		/// Generates electronic transfusion exception
		/// BR_69.21 CR 2942
		/// </summary>
		/// <param name="drPatientTransfusion"></param>
		/// <param name="patient"></param>
		/// <param name="bloodUnit"></param>
		/// <param name="comment"></param>
		public void GenerateExceptionElectronicTransfusion(DataRow drPatientTransfusion, BOL.Patient patient, BOL.BloodUnit bloodUnit, string comment)
		{
			DataTable dtOriginalTransfusionRecord = DAL.PatientTransfusion.GetPatientTransfusionRecord(patient.PatientGuid, bloodUnit.BloodUnitGuid);

			if (dtOriginalTransfusionRecord.Rows.Count > 0)
			{
				DataTable dtExceptionElectronicTransfusion = new DataTable(TABLE.ExceptionElectronicTransfusion.TableName);
				dtExceptionElectronicTransfusion.Columns.Add(TABLE.ExceptionElectronicTransfusion.BedsideVerificationDate, typeof(DateTime));
				dtExceptionElectronicTransfusion.Columns.Add(TABLE.ExceptionElectronicTransfusion.BedsideVerificationUnitManualEntryReason);
				dtExceptionElectronicTransfusion.Columns.Add(TABLE.ExceptionElectronicTransfusion.DivisionCode);
				dtExceptionElectronicTransfusion.Columns.Add(TABLE.ExceptionElectronicTransfusion.ExceptionDate, typeof(DateTime));
				dtExceptionElectronicTransfusion.Columns.Add(TABLE.ExceptionElectronicTransfusion.ExceptionElectronicTransfusionGuid, typeof(Guid));
				dtExceptionElectronicTransfusion.Columns.Add(TABLE.ExceptionElectronicTransfusion.ExceptionTypeId, typeof(int));
				dtExceptionElectronicTransfusion.Columns.Add(TABLE.ExceptionElectronicTransfusion.ExceptionUser);
				dtExceptionElectronicTransfusion.Columns.Add(TABLE.ExceptionElectronicTransfusion.InterruptionIndicator, typeof(bool));
				dtExceptionElectronicTransfusion.Columns.Add(TABLE.ExceptionElectronicTransfusion.PatientTransfusionGuid, typeof(Guid));
				dtExceptionElectronicTransfusion.Columns.Add(TABLE.ExceptionElectronicTransfusion.ReactionIndicator, typeof(bool));
				dtExceptionElectronicTransfusion.Columns.Add(TABLE.ExceptionElectronicTransfusion.ReactionSymptoms);
				dtExceptionElectronicTransfusion.Columns.Add(TABLE.ExceptionElectronicTransfusion.TransfusedVolume, typeof(int));
				dtExceptionElectronicTransfusion.Columns.Add(TABLE.ExceptionElectronicTransfusion.TransfusionEndDateTime, typeof(DateTime));
				dtExceptionElectronicTransfusion.Columns.Add(TABLE.ExceptionElectronicTransfusion.Transfusionist1Id);
				dtExceptionElectronicTransfusion.Columns.Add(TABLE.ExceptionElectronicTransfusion.Transfusionist2Id);
				dtExceptionElectronicTransfusion.Columns.Add(TABLE.ExceptionElectronicTransfusion.TransfusionLocation);
				dtExceptionElectronicTransfusion.Columns.Add(TABLE.ExceptionElectronicTransfusion.TransfusionStartDateTime, typeof(DateTime));
				dtExceptionElectronicTransfusion.Columns.Add(TABLE.ExceptionElectronicTransfusion.PatientName);
				dtExceptionElectronicTransfusion.Columns.Add(TABLE.ExceptionElectronicTransfusion.PatientId);
				dtExceptionElectronicTransfusion.Columns.Add(TABLE.ExceptionElectronicTransfusion.UnitId);
				dtExceptionElectronicTransfusion.Columns.Add(TABLE.ExceptionElectronicTransfusion.UnitProductCode);
				dtExceptionElectronicTransfusion.Columns.Add(TABLE.ExceptionElectronicTransfusion.PatientTransfusionComment);

				this.ExceptionData = dtExceptionElectronicTransfusion.NewRow();
				this.ExceptionData[TABLE.ExceptionElectronicTransfusion.ExceptionElectronicTransfusionGuid] = Guid.NewGuid();
				this.ExceptionData[TABLE.ExceptionElectronicTransfusion.ExceptionTypeId] = this.ExceptionType;
				this.ExceptionData[TABLE.ExceptionElectronicTransfusion.DivisionCode] = Common.LogonUser.LogonUserDivisionCode;
				this.ExceptionData[TABLE.ExceptionElectronicTransfusion.ExceptionDate] = this.ExceptionDate;
				this.ExceptionData[TABLE.ExceptionElectronicTransfusion.ExceptionUser] = "Electronic Entry";
				this.ExceptionData[TABLE.ExceptionElectronicTransfusion.PatientTransfusionGuid] = drPatientTransfusion[TABLE.PatientTransfusion.PatientTransfusionGuid];
				this.ExceptionData[TABLE.ExceptionElectronicTransfusion.PatientName] = patient.LastName + ", " + patient.FirstName + " " + patient.MiddleInitial;
				this.ExceptionData[TABLE.ExceptionElectronicTransfusion.PatientId] = patient.DisplayPatientID;
				this.ExceptionData[TABLE.ExceptionElectronicTransfusion.UnitId]	= bloodUnit.EyeReadableUnitId;
				this.ExceptionData[TABLE.ExceptionElectronicTransfusion.UnitProductCode]= bloodUnit.ProductCode;
				this.ExceptionData[TABLE.ExceptionElectronicTransfusion.PatientTransfusionComment]= comment;

				if (drPatientTransfusion.IsNull(TABLE.PatientTransfusion.BedsideVerificationDate) == false)
				{
					if (drPatientTransfusion[TABLE.PatientTransfusion.BedsideVerificationDate].Equals(dtOriginalTransfusionRecord.Rows[0][TABLE.PatientTransfusion.BedsideVerificationDate]))
					{
						this.ExceptionData[TABLE.ExceptionElectronicTransfusion.BedsideVerificationDate] = DBNull.Value;
					}
					else
					{
						this.ExceptionData[TABLE.ExceptionElectronicTransfusion.BedsideVerificationDate] = drPatientTransfusion[TABLE.PatientTransfusion.BedsideVerificationDate];
					}
				}

				if (drPatientTransfusion.IsNull(TABLE.PatientTransfusion.BedsideVerificationUnitManualEntryReason) == false)
				{
					if (drPatientTransfusion[TABLE.PatientTransfusion.BedsideVerificationUnitManualEntryReason].Equals(dtOriginalTransfusionRecord.Rows[0][TABLE.PatientTransfusion.BedsideVerificationUnitManualEntryReason]))
					{
						this.ExceptionData[TABLE.ExceptionElectronicTransfusion.BedsideVerificationUnitManualEntryReason] = DBNull.Value;
					}
					else
					{
						this.ExceptionData[TABLE.ExceptionElectronicTransfusion.BedsideVerificationUnitManualEntryReason] = drPatientTransfusion[TABLE.PatientTransfusion.BedsideVerificationUnitManualEntryReason];
					}
				}

				if (drPatientTransfusion.IsNull(TABLE.PatientTransfusion.InterruptionIndicator) == false)
				{
					if (drPatientTransfusion[TABLE.PatientTransfusion.InterruptionIndicator].Equals(dtOriginalTransfusionRecord.Rows[0][TABLE.PatientTransfusion.InterruptionIndicator]))
					{
						this.ExceptionData[TABLE.ExceptionElectronicTransfusion.InterruptionIndicator] = DBNull.Value;
					}
					else
					{
						this.ExceptionData[TABLE.ExceptionElectronicTransfusion.InterruptionIndicator] = drPatientTransfusion[TABLE.PatientTransfusion.InterruptionIndicator];
					}
				}

				if (drPatientTransfusion.IsNull(TABLE.PatientTransfusion.ReactionIndicator) == false)
				{
					if (drPatientTransfusion[TABLE.PatientTransfusion.ReactionIndicator].Equals(dtOriginalTransfusionRecord.Rows[0][TABLE.PatientTransfusion.ReactionIndicator]))
					{
						this.ExceptionData[TABLE.ExceptionElectronicTransfusion.ReactionIndicator] = DBNull.Value;
					}
					else
					{
						this.ExceptionData[TABLE.ExceptionElectronicTransfusion.ReactionIndicator] = drPatientTransfusion[TABLE.PatientTransfusion.ReactionIndicator];
					}
				}

				if (drPatientTransfusion.IsNull(TABLE.PatientTransfusion.ReactionSymptoms) == false)
				{
					if (drPatientTransfusion[TABLE.PatientTransfusion.ReactionSymptoms].Equals(dtOriginalTransfusionRecord.Rows[0][TABLE.PatientTransfusion.ReactionSymptoms]))
					{
						this.ExceptionData[TABLE.ExceptionElectronicTransfusion.ReactionSymptoms] = DBNull.Value;
					}
					else
					{
						this.ExceptionData[TABLE.ExceptionElectronicTransfusion.ReactionSymptoms] = drPatientTransfusion[TABLE.PatientTransfusion.ReactionSymptoms];
					}
				}

				if (drPatientTransfusion.IsNull(TABLE.PatientTransfusion.TransfusedVolume) == false)
				{
					if (drPatientTransfusion[TABLE.PatientTransfusion.TransfusedVolume].Equals(dtOriginalTransfusionRecord.Rows[0][TABLE.PatientTransfusion.TransfusedVolume]))
					{
						this.ExceptionData[TABLE.ExceptionElectronicTransfusion.TransfusedVolume] = DBNull.Value;
					}
					else
					{
						this.ExceptionData[TABLE.ExceptionElectronicTransfusion.TransfusedVolume] = drPatientTransfusion[TABLE.PatientTransfusion.TransfusedVolume];
					}
				}

				if (drPatientTransfusion.IsNull(TABLE.PatientTransfusion.TransfusionEndDateTime) == false)
				{
					if (drPatientTransfusion[TABLE.PatientTransfusion.TransfusionEndDateTime].Equals(dtOriginalTransfusionRecord.Rows[0][TABLE.PatientTransfusion.TransfusionEndDateTime]))
					{
						this.ExceptionData[TABLE.ExceptionElectronicTransfusion.TransfusionEndDateTime] = DBNull.Value;
					}
					else
					{
						this.ExceptionData[TABLE.ExceptionElectronicTransfusion.TransfusionEndDateTime] = drPatientTransfusion[TABLE.PatientTransfusion.TransfusionEndDateTime];
					}
				}

				if (drPatientTransfusion.IsNull(TABLE.PatientTransfusion.Transfusionist1Id) == false)
				{
					if (drPatientTransfusion[TABLE.PatientTransfusion.Transfusionist1Id].Equals(dtOriginalTransfusionRecord.Rows[0][TABLE.PatientTransfusion.Transfusionist1Id]))
					{
						this.ExceptionData[TABLE.ExceptionElectronicTransfusion.Transfusionist1Id] = DBNull.Value;
					}
					else
					{
						this.ExceptionData[TABLE.ExceptionElectronicTransfusion.Transfusionist1Id] = drPatientTransfusion[TABLE.PatientTransfusion.Transfusionist1Id];
					}
				}

				if (drPatientTransfusion.IsNull(TABLE.PatientTransfusion.Transfusionist2Id) == false)
				{
					if (drPatientTransfusion[TABLE.PatientTransfusion.Transfusionist2Id].Equals(dtOriginalTransfusionRecord.Rows[0][TABLE.PatientTransfusion.Transfusionist2Id]))
					{
						this.ExceptionData[TABLE.ExceptionElectronicTransfusion.Transfusionist2Id] = DBNull.Value;
					}
					else
					{
						this.ExceptionData[TABLE.ExceptionElectronicTransfusion.Transfusionist2Id] = drPatientTransfusion[TABLE.PatientTransfusion.Transfusionist2Id];
					}
				}

				if (drPatientTransfusion.IsNull(TABLE.PatientTransfusion.TransfusionLocation) == false)
				{
					if (drPatientTransfusion[TABLE.PatientTransfusion.TransfusionLocation].Equals(dtOriginalTransfusionRecord.Rows[0][TABLE.PatientTransfusion.TransfusionLocation]))
					{
						this.ExceptionData[TABLE.ExceptionElectronicTransfusion.TransfusionLocation] = DBNull.Value;
					}
					else
					{
						this.ExceptionData[TABLE.ExceptionElectronicTransfusion.TransfusionLocation] = drPatientTransfusion[TABLE.PatientTransfusion.TransfusionLocation];
					}
				}

				if (drPatientTransfusion.IsNull(TABLE.PatientTransfusion.TransfusionStartDateTime) == false)
				{
					if (drPatientTransfusion[TABLE.PatientTransfusion.TransfusionStartDateTime].Equals(dtOriginalTransfusionRecord.Rows[0][TABLE.PatientTransfusion.TransfusionStartDateTime]))
					{
						this.ExceptionData[TABLE.ExceptionElectronicTransfusion.TransfusionStartDateTime] = DBNull.Value;
					}
					else
					{
						this.ExceptionData[TABLE.ExceptionElectronicTransfusion.TransfusionStartDateTime] = drPatientTransfusion[TABLE.PatientTransfusion.TransfusionStartDateTime];
					}
				}
			}
		}

		///<Developers>
		///	<Developer>Unknown</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>10/7/2004</CreationDate>
		///<TestCases>
		///
		///<Case type="0" testid ="2307"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>DataSet</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="2316"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Get the DataSet that is used as a data source for ExceptionReportMain
		/// It will be used to suppress some of the exception subreports (turn on or off certain exception reports)
		/// </summary>
		/// <returns></returns>
		public static DataSet GetExceptionReportMain()
		{
			DataSet ds = new DataSet("ExceptionReportMain");
			DataTable dt = new DataTable("ExceptionReportMain");
			
			ds.Tables.Add(dt);

			dt.Columns.Add(string.Concat(TABLE.ExceptionLabelMismatch.TableName, "Suppress"), typeof(bool));
			dt.Columns.Add(string.Concat(TABLE.ExceptionPatientAboRh.TableName, "Suppress"), typeof(bool));
			dt.Columns.Add(string.Concat(TABLE.ExceptionPatientAboRhDiscrepancy.TableName, "Suppress"), typeof(bool));
			dt.Columns.Add(string.Concat(TABLE.ExceptionPatientAntigenDiscrepancy.TableName, "Suppress"), typeof(bool));
			dt.Columns.Add(string.Concat(TABLE.ExceptionPatientTest.TableName, "Suppress"), typeof(bool));
			dt.Columns.Add(string.Concat(TABLE.ExceptionPatientTestInactivated.TableName, "Suppress"), typeof(bool));
			dt.Columns.Add(string.Concat(TABLE.ExceptionQc.TableName, "Suppress"), typeof(bool));
			dt.Columns.Add(string.Concat(TABLE.ExceptionQcInactivation.TableName, "Suppress"), typeof(bool));
			dt.Columns.Add(string.Concat(TABLE.ExceptionQcNotPerformed.TableName, "Suppress"), typeof(bool));
			dt.Columns.Add(string.Concat(TABLE.ExceptionQuarantine.TableName, "Suppress"), typeof(bool));
			dt.Columns.Add(string.Concat(TABLE.ExceptionReflexTest.TableName, "Suppress"), typeof(bool));
			dt.Columns.Add(string.Concat(TABLE.ExceptionSpecimen.TableName, "Suppress"), typeof(bool));
			dt.Columns.Add(string.Concat(TABLE.ExceptionUnitAboRh.TableName, "Suppress"), typeof(bool));
			dt.Columns.Add(string.Concat(TABLE.ExceptionUnitModification.TableName, "Suppress"), typeof(bool));
			dt.Columns.Add(string.Concat(TABLE.ExceptionUnitSelection.TableName, "Suppress"), typeof(bool));
			dt.Columns.Add(string.Concat(TABLE.ExceptionExpiredUnitReceived.TableName, "Suppress"), typeof(bool));
			dt.Columns.Add(string.Concat(TABLE.ExceptionCautionTag.TableName, "Suppress"), typeof(bool));
			dt.Columns.Add(string.Concat(TABLE.ExceptionTrwSpecimen.TableName, "Suppress"), typeof(bool));
			dt.Columns.Add(string.Concat(TABLE.ExceptionTransfusionReactionWorkup.TableName, "Suppress"), typeof(bool));
			dt.Columns.Add(string.Concat(TABLE.ExceptionAntigenTest.TableName, "Suppress"), typeof(bool));
			dt.Columns.Add("ExceptionUnitTestSuppress", typeof(bool));
			dt.Columns.Add(string.Concat(TABLE.ExceptionReturnedUnit.TableName, "Suppress"), typeof(bool));
			dt.Columns.Add(string.Concat(TABLE.ExceptionTransfusion.TableName, "Suppress"), typeof(bool));
			dt.Columns.Add(string.Concat(TABLE.ExceptionExpiredTask.TableName, "Suppress"), typeof(bool));
			dt.Columns.Add(string.Concat(TABLE.ExceptionEmergencyIssueUnitTest.TableName, "Suppress"), typeof(bool));
			dt.Columns.Add(string.Concat(TABLE.ExceptionUnitIssue.TableName, "Suppress"), typeof(bool));
			dt.Columns.Add("ExceptionReportUnitSupplySuppress", typeof(bool));
			dt.Columns.Add(string.Concat(TABLE.ExceptionPatientRestriction.TableName, "Suppress"), typeof(bool));
			dt.Columns.Add(string.Concat(TABLE.ExceptionEmergencyIssueOrder.TableName, "Suppress"), typeof(bool));
			dt.Columns.Add(string.Concat(TABLE.ExceptionRemoveFinalUnitStatus.TableName, "Suppress"), typeof(bool));
			dt.Columns.Add(string.Concat(TABLE.ExceptionInconclusiveCrossmatch.TableName, "Suppress"), typeof(bool));
			dt.Columns.Add(string.Concat(TABLE.ExceptionElectronicTransfusion.TableName, "Suppress"), typeof(bool));
            
			DataRow dr = dt.NewRow();
			dt.Rows.Add(dr);

			return ds;
		}

		///<Developers>
		///	<Developer>Rob Heiberger</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/31/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6636"> 
		///		<ExpectedInput>N/A</ExpectedInput>
		///		<ExpectedOutput>N/A</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="6637"> 
		///		<ExpectedInput>N/A</ExpectedInput>
		///		<ExpectedOutput>N/A</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// GenerateAndPrintReport
		/// </summary>
        protected override VbecsReportPrinter createReportPrinter()
		{

            DataSet dsExceptionReportQC = BOL.ExceptionReport.GetExceptionReport(this.StartDate, this.EndDate, Common.ExceptionReportType.ExceptionReportQC);
            _exceptionReports.Add(Common.ExceptionReportType.ExceptionReportQC, dsExceptionReportQC);

            DataSet dsExceptionReportQCInactivated = BOL.ExceptionReport.GetExceptionReport(this.StartDate, this.EndDate, Common.ExceptionReportType.ExceptionReportQCInactivated);
            _exceptionReports.Add(Common.ExceptionReportType.ExceptionReportQCInactivated, dsExceptionReportQCInactivated);

            DataSet dsExceptionReportQCNotPerformed = BOL.ExceptionReport.GetExceptionReport(this.StartDate, this.EndDate, Common.ExceptionReportType.ExceptionReportQCNotPerformed);
            _exceptionReports.Add(Common.ExceptionReportType.ExceptionReportQCNotPerformed, dsExceptionReportQCNotPerformed);

            DataSet dsExceptionReportQuarantine = BOL.ExceptionReport.GetExceptionReport(this.StartDate, this.EndDate, Common.ExceptionReportType.ExceptionReportQuarantine);
            _exceptionReports.Add(Common.ExceptionReportType.ExceptionReportQuarantine, dsExceptionReportQuarantine);

            DataSet dsExceptionReportReflexTest = BOL.ExceptionReport.GetExceptionReport(this.StartDate, this.EndDate, Common.ExceptionReportType.ExceptionReportReflexTest);
            _exceptionReports.Add(Common.ExceptionReportType.ExceptionReportReflexTest, dsExceptionReportReflexTest);

            DataSet dsExceptionReportSpecimen = BOL.ExceptionReport.GetExceptionReport(this.StartDate, this.EndDate, Common.ExceptionReportType.ExceptionReportSpecimen);
            _exceptionReports.Add(Common.ExceptionReportType.ExceptionReportSpecimen, dsExceptionReportSpecimen);

            DataSet dsExceptionReportPatientAboRh = BOL.ExceptionReport.GetExceptionReport(this.StartDate, this.EndDate, Common.ExceptionReportType.ExceptionReportPatientAboRh);
            _exceptionReports.Add(Common.ExceptionReportType.ExceptionReportPatientAboRh, dsExceptionReportPatientAboRh);

            DataSet dsExceptionReportPatientAboRhDiscrepancy = BOL.ExceptionReport.GetExceptionReport(this.StartDate, this.EndDate, Common.ExceptionReportType.ExceptionReportPatientAboRhDiscrepancy);
            _exceptionReports.Add(Common.ExceptionReportType.ExceptionReportPatientAboRhDiscrepancy, dsExceptionReportPatientAboRhDiscrepancy);

            DataSet dsExceptionReportPatientAntigenDiscrepancy = BOL.ExceptionReport.GetExceptionReport(this.StartDate, this.EndDate, Common.ExceptionReportType.ExceptionReportPatientAntigenDiscrepancy);
            _exceptionReports.Add(Common.ExceptionReportType.ExceptionReportPatientAntigenDiscrepancy, dsExceptionReportPatientAntigenDiscrepancy);

            DataSet dsExceptionReportPatientTest = BOL.ExceptionReport.GetExceptionReport(this.StartDate, this.EndDate, Common.ExceptionReportType.ExceptionReportPatientTest);
            _exceptionReports.Add(Common.ExceptionReportType.ExceptionReportPatientTest, dsExceptionReportPatientTest);

            DataSet dsExceptionReportPatientTestInactivated = BOL.ExceptionReport.GetExceptionReport(this.StartDate, this.EndDate, Common.ExceptionReportType.ExceptionReportPatientTestInactivated);
            _exceptionReports.Add(Common.ExceptionReportType.ExceptionReportPatientTestInactivated, dsExceptionReportPatientTestInactivated);

            DataSet dsExceptionReportUnitAboRh = BOL.ExceptionReport.GetExceptionReport(this.StartDate, this.EndDate, Common.ExceptionReportType.ExceptionReportUnitAboRh);
            _exceptionReports.Add(Common.ExceptionReportType.ExceptionReportUnitAboRh, dsExceptionReportUnitAboRh);

            DataSet dsExceptionReportUnitModification = BOL.ExceptionReport.GetExceptionReport(this.StartDate, this.EndDate, Common.ExceptionReportType.ExceptionReportUnitModification);
            _exceptionReports.Add(Common.ExceptionReportType.ExceptionReportUnitModification, dsExceptionReportUnitModification);

            DataSet dsExceptionReportUnitSelection = BOL.ExceptionReport.GetExceptionReport(this.StartDate, this.EndDate, Common.ExceptionReportType.ExceptionReportUnitSelection);
            _exceptionReports.Add(Common.ExceptionReportType.ExceptionReportUnitSelection, dsExceptionReportUnitSelection);

            DataSet dsExceptionReportLabelMismatch = BOL.ExceptionReport.GetExceptionReport(this.StartDate, this.EndDate, Common.ExceptionReportType.ExceptionReportLabelMismatch);
            _exceptionReports.Add(Common.ExceptionReportType.ExceptionReportLabelMismatch, dsExceptionReportLabelMismatch);

            DataSet dsExceptionReportCautionTagBTRFPrinting = BOL.ExceptionReport.GetExceptionReport(this.StartDate, this.EndDate, Common.ExceptionReportType.ExceptionReportCautionTagBTRFPrinting);
            _exceptionReports.Add(Common.ExceptionReportType.ExceptionReportCautionTagBTRFPrinting, dsExceptionReportCautionTagBTRFPrinting);

            DataSet dsExceptionReportTrwSpecimen = BOL.ExceptionReport.GetExceptionReport(this.StartDate, this.EndDate, Common.ExceptionReportType.ExceptionReportTrwSpecimen);
            _exceptionReports.Add(Common.ExceptionReportType.ExceptionReportTrwSpecimen, dsExceptionReportTrwSpecimen);

            DataSet dsExceptionReportTransfusionReactionWorkup = BOL.ExceptionReport.GetExceptionReport(this.StartDate, this.EndDate, Common.ExceptionReportType.ExceptionReportTransfusionReactionWorkup);
            _exceptionReports.Add(Common.ExceptionReportType.ExceptionReportTransfusionReactionWorkup, dsExceptionReportTransfusionReactionWorkup);

            DataSet dsExceptionReportAntigenTest = BOL.ExceptionReport.GetExceptionReport(this.StartDate, this.EndDate, Common.ExceptionReportType.ExceptionReportAntigenTest);
            _exceptionReports.Add(Common.ExceptionReportType.ExceptionReportAntigenTest, dsExceptionReportAntigenTest);

            DataSet dsExceptionReportUnitTest = BOL.ExceptionReport.GetExceptionReport(this.StartDate, this.EndDate, Common.ExceptionReportType.ExceptionReportUnitTest);
            _exceptionReports.Add(Common.ExceptionReportType.ExceptionReportUnitTest, dsExceptionReportUnitTest);

            DataSet dsExceptionReportReturnIssuedUnit = BOL.ExceptionReport.GetExceptionReport(this.StartDate, this.EndDate, Common.ExceptionReportType.ExceptionReportReturnIssuedUnit);
            _exceptionReports.Add(Common.ExceptionReportType.ExceptionReportReturnIssuedUnit, dsExceptionReportReturnIssuedUnit);

            DataSet dsExceptionReportTransfusion = BOL.ExceptionReport.GetExceptionReport(this.StartDate, this.EndDate, Common.ExceptionReportType.ExceptionReportTransfusion);
            _exceptionReports.Add(Common.ExceptionReportType.ExceptionReportTransfusion, dsExceptionReportTransfusion);

            DataSet dsExceptionReportExpiredUnit = BOL.ExceptionReport.GetExceptionReport(this.StartDate, this.EndDate, Common.ExceptionReportType.ExceptionReportExpiredUnitReceived);
            _exceptionReports.Add(Common.ExceptionReportType.ExceptionReportExpiredUnitReceived, dsExceptionReportExpiredUnit);

            DataSet dsExceptionReportUnitSupply = BOL.ExceptionReport.GetExceptionReport(this.StartDate, this.EndDate, Common.ExceptionReportType.ExceptionReportUnitSupply);
            _exceptionReports.Add(Common.ExceptionReportType.ExceptionReportUnitSupply, dsExceptionReportUnitSupply);

            DataSet dsExceptionReportUnitIssue = BOL.ExceptionReport.GetExceptionReport(this.StartDate, this.EndDate, Common.ExceptionReportType.ExceptionReportUnitIssue);
            _exceptionReports.Add(Common.ExceptionReportType.ExceptionReportUnitIssue, dsExceptionReportUnitIssue);

            DataSet dsExceptionReportPatientRestriction = BOL.ExceptionReport.GetExceptionReport(this.StartDate, this.EndDate, Common.ExceptionReportType.ExceptionReportPatientRestriction);
            _exceptionReports.Add(Common.ExceptionReportType.ExceptionReportPatientRestriction, dsExceptionReportPatientRestriction);

            DataSet dsExceptionReportEmergencyIssueOrder = BOL.ExceptionReport.GetExceptionReport(this.StartDate, this.EndDate, Common.ExceptionReportType.ExceptionReportEmergencyIssueOrder);
            _exceptionReports.Add(Common.ExceptionReportType.ExceptionReportEmergencyIssueOrder, dsExceptionReportEmergencyIssueOrder);

            DataSet dsExceptionReportEmergencyIssueUnitTest = BOL.ExceptionReport.GetExceptionReport(this.StartDate, this.EndDate, Common.ExceptionReportType.ExceptionReportUnitERIssuedTestingProblem);
            _exceptionReports.Add(Common.ExceptionReportType.ExceptionReportUnitERIssuedTestingProblem, dsExceptionReportEmergencyIssueUnitTest);

            DataSet dsExceptionReportExpiredOrder = BOL.ExceptionReport.GetExceptionReport(this.StartDate, this.EndDate, Common.ExceptionReportType.ExceptionReportExpiredTaskProcessed);
            _exceptionReports.Add(Common.ExceptionReportType.ExceptionReportExpiredTaskProcessed, dsExceptionReportExpiredOrder);

            DataSet dsExceptionReportRemoveFinalUnitStatus = BOL.ExceptionReport.GetExceptionReport(this.StartDate, this.EndDate, Common.ExceptionReportType.ExceptionReportRemoveFinalUnitStatus);
            _exceptionReports.Add(Common.ExceptionReportType.ExceptionReportRemoveFinalUnitStatus, dsExceptionReportRemoveFinalUnitStatus);

            DataSet dsExceptionReportInconclusiveCrossmatch = BOL.ExceptionReport.GetExceptionReport(this.StartDate, this.EndDate, Common.ExceptionReportType.ExceptionReportCrossmatch);
            _exceptionReports.Add(Common.ExceptionReportType.ExceptionReportCrossmatch, dsExceptionReportInconclusiveCrossmatch);

            DataSet dsExceptionReportElectronicTransfusion = BOL.ExceptionReport.GetExceptionReport(this.StartDate, this.EndDate, Common.ExceptionReportType.ExceptionReportElectronicTransfusion);
            _exceptionReports.Add(Common.ExceptionReportType.ExceptionReportElectronicTransfusion, dsExceptionReportElectronicTransfusion);

            DataSet dsExceptionReportMain = BOL.ExceptionReport.GetExceptionReportMain();
            _exceptionReports.Add(Common.ExceptionReportType.ExceptionReportMain, dsExceptionReportMain);

            if (dsExceptionReportQC.Tables.Count > 0)
            {
                if (dsExceptionReportQC.Tables[0].Rows.Count > 0)
                {
                    dsExceptionReportMain.Tables[0].Rows[0][Common.VbecsTables.ExceptionQc.TableName + "Suppress"] = false;
                }
                else
                {
                    dsExceptionReportMain.Tables[0].Rows[0][Common.VbecsTables.ExceptionQc.TableName + "Suppress"] = true;
                }
            }


            if (dsExceptionReportQCInactivated.Tables.Count > 0)
            {
                if (dsExceptionReportQCInactivated.Tables[0].Rows.Count > 0)
                {
                    dsExceptionReportMain.Tables[0].Rows[0][Common.VbecsTables.ExceptionQcInactivation.TableName + "Suppress"] = false;
                }
                else
                {
                    dsExceptionReportMain.Tables[0].Rows[0][Common.VbecsTables.ExceptionQcInactivation.TableName + "Suppress"] = true;
                }
            }
            if (dsExceptionReportQCNotPerformed.Tables.Count > 0)
            {
                if (dsExceptionReportQCNotPerformed.Tables[0].Rows.Count > 0)
                {
                    dsExceptionReportMain.Tables[0].Rows[0][Common.VbecsTables.ExceptionQcNotPerformed.TableName + "Suppress"] = false;
                }
                else
                {
                    dsExceptionReportMain.Tables[0].Rows[0][Common.VbecsTables.ExceptionQcNotPerformed.TableName + "Suppress"] = true;
                }
            }

            if (dsExceptionReportQuarantine.Tables.Count > 0)
            {
                if (dsExceptionReportQuarantine.Tables[0].Rows.Count > 0)
                {
                    dsExceptionReportMain.Tables[0].Rows[0][Common.VbecsTables.ExceptionQuarantine.TableName + "Suppress"] = false;
                }
                else
                {
                    dsExceptionReportMain.Tables[0].Rows[0][Common.VbecsTables.ExceptionQuarantine.TableName + "Suppress"] = true;
                }
            }

            if (dsExceptionReportReflexTest.Tables.Count > 0)
            {
                if (dsExceptionReportReflexTest.Tables[0].Rows.Count > 0)
                {
                    dsExceptionReportMain.Tables[0].Rows[0][Common.VbecsTables.ExceptionReflexTest.TableName + "Suppress"] = false;
                }
                else
                {
                    dsExceptionReportMain.Tables[0].Rows[0][Common.VbecsTables.ExceptionReflexTest.TableName + "Suppress"] = true;
                }
            }

            if (dsExceptionReportSpecimen.Tables.Count > 0)
            {
                if (dsExceptionReportSpecimen.Tables[0].Rows.Count > 0)
                {
                    dsExceptionReportMain.Tables[0].Rows[0][Common.VbecsTables.ExceptionSpecimen.TableName + "Suppress"] = false;
                }
                else
                {
                    dsExceptionReportMain.Tables[0].Rows[0][Common.VbecsTables.ExceptionSpecimen.TableName + "Suppress"] = true;
                }
            }

            if (dsExceptionReportPatientAboRh.Tables.Count > 0)
            {
                if (dsExceptionReportPatientAboRh.Tables[0].Rows.Count > 0)
                {
                    dsExceptionReportMain.Tables[0].Rows[0][Common.VbecsTables.ExceptionPatientAboRh.TableName + "Suppress"] = false;
                }
                else
                {
                    dsExceptionReportMain.Tables[0].Rows[0][Common.VbecsTables.ExceptionPatientAboRh.TableName + "Suppress"] = true;
                }
            }

            if (dsExceptionReportPatientAboRhDiscrepancy.Tables.Count > 0)
            {
                if (dsExceptionReportPatientAboRhDiscrepancy.Tables[0].Rows.Count > 0)
                {
                    dsExceptionReportMain.Tables[0].Rows[0][Common.VbecsTables.ExceptionPatientAboRhDiscrepancy.TableName + "Suppress"] = false;
                }
                else
                {
                    dsExceptionReportMain.Tables[0].Rows[0][Common.VbecsTables.ExceptionPatientAboRhDiscrepancy.TableName + "Suppress"] = true;
                }
            }

            if (dsExceptionReportPatientTest.Tables.Count > 0)
            {
                if (dsExceptionReportPatientTest.Tables[0].Rows.Count > 0)
                {
                    dsExceptionReportMain.Tables[0].Rows[0][Common.VbecsTables.ExceptionPatientTest.TableName + "Suppress"] = false;
                }
                else
                {
                    dsExceptionReportMain.Tables[0].Rows[0][Common.VbecsTables.ExceptionPatientTest.TableName + "Suppress"] = true;
                }
            }

            if (dsExceptionReportPatientTestInactivated.Tables.Count > 0)
            {
                if (dsExceptionReportPatientTestInactivated.Tables[0].Rows.Count > 0)
                {
                    dsExceptionReportMain.Tables[0].Rows[0][Common.VbecsTables.ExceptionPatientTestInactivated.TableName + "Suppress"] = false;
                }
                else
                {
                    dsExceptionReportMain.Tables[0].Rows[0][Common.VbecsTables.ExceptionPatientTestInactivated.TableName + "Suppress"] = true;
                }
            }

            if (dsExceptionReportUnitAboRh.Tables.Count > 0)
            {
                if (dsExceptionReportUnitAboRh.Tables[0].Rows.Count > 0)
                {
                    dsExceptionReportMain.Tables[0].Rows[0][Common.VbecsTables.ExceptionUnitAboRh.TableName + "Suppress"] = false;
                }
                else
                {
                    dsExceptionReportMain.Tables[0].Rows[0][Common.VbecsTables.ExceptionUnitAboRh.TableName + "Suppress"] = true;
                }
            }

            if (dsExceptionReportPatientAntigenDiscrepancy.Tables.Count > 0)
            {
                if (dsExceptionReportPatientAntigenDiscrepancy.Tables[0].Rows.Count > 0)
                {
                    dsExceptionReportMain.Tables[0].Rows[0][Common.VbecsTables.ExceptionPatientAntigenDiscrepancy.TableName + "Suppress"] = false;
                }
                else
                {
                    dsExceptionReportMain.Tables[0].Rows[0][Common.VbecsTables.ExceptionPatientAntigenDiscrepancy.TableName + "Suppress"] = true;
                }
            }

            if (dsExceptionReportUnitModification.Tables.Count > 0)
            {
                if (dsExceptionReportUnitModification.Tables[0].Rows.Count > 0)
                {
                    dsExceptionReportMain.Tables[0].Rows[0][Common.VbecsTables.ExceptionUnitModification.TableName + "Suppress"] = false;
                }
                else
                {
                    dsExceptionReportMain.Tables[0].Rows[0][Common.VbecsTables.ExceptionUnitModification.TableName + "Suppress"] = true;
                }
            }

            if (dsExceptionReportUnitSelection.Tables.Count > 0)
            {
                if (dsExceptionReportUnitSelection.Tables[0].Rows.Count > 0)
                {
                    dsExceptionReportMain.Tables[0].Rows[0][Common.VbecsTables.ExceptionUnitSelection.TableName + "Suppress"] = false;
                }
                else
                {
                    dsExceptionReportMain.Tables[0].Rows[0][Common.VbecsTables.ExceptionUnitSelection.TableName + "Suppress"] = true;
                }
            }

            if (dsExceptionReportLabelMismatch.Tables.Count > 0)
            {
                if (dsExceptionReportLabelMismatch.Tables[0].Rows.Count > 0)
                {
                    dsExceptionReportMain.Tables[0].Rows[0][Common.VbecsTables.ExceptionLabelMismatch.TableName + "Suppress"] = false;
                }
                else
                {
                    dsExceptionReportMain.Tables[0].Rows[0][Common.VbecsTables.ExceptionLabelMismatch.TableName + "Suppress"] = true;
                }
            }

            if (dsExceptionReportCautionTagBTRFPrinting.Tables.Count > 0)
            {
                if (dsExceptionReportCautionTagBTRFPrinting.Tables[0].Rows.Count > 0)
                {
                    dsExceptionReportMain.Tables[0].Rows[0][Common.VbecsTables.ExceptionCautionTag.TableName + "Suppress"] = false;
                }
                else
                {
                    dsExceptionReportMain.Tables[0].Rows[0][Common.VbecsTables.ExceptionCautionTag.TableName + "Suppress"] = true;
                }
            }

            if (dsExceptionReportTrwSpecimen.Tables.Count > 0)
            {
                if (dsExceptionReportTrwSpecimen.Tables[0].Rows.Count > 0)
                {
                    dsExceptionReportMain.Tables[0].Rows[0][Common.VbecsTables.ExceptionTrwSpecimen.TableName + "Suppress"] = false;
                }
                else
                {
                    dsExceptionReportMain.Tables[0].Rows[0][Common.VbecsTables.ExceptionTrwSpecimen.TableName + "Suppress"] = true;
                }
            }

            if (dsExceptionReportTransfusionReactionWorkup.Tables.Count > 0)
            {
                if (dsExceptionReportTransfusionReactionWorkup.Tables[0].Rows.Count > 0)
                {
                    dsExceptionReportMain.Tables[0].Rows[0][Common.VbecsTables.ExceptionTransfusionReactionWorkup.TableName + "Suppress"] = false;
                }
                else
                {
                    dsExceptionReportMain.Tables[0].Rows[0][Common.VbecsTables.ExceptionTransfusionReactionWorkup.TableName + "Suppress"] = true;
                }
            }

            if (dsExceptionReportAntigenTest.Tables.Count > 0)
            {
                if (dsExceptionReportAntigenTest.Tables[0].Rows.Count > 0)
                {
                    dsExceptionReportMain.Tables[0].Rows[0][Common.VbecsTables.ExceptionAntigenTest.TableName + "Suppress"] = false;
                }
                else
                {
                    dsExceptionReportMain.Tables[0].Rows[0][Common.VbecsTables.ExceptionAntigenTest.TableName + "Suppress"] = true;
                }
            }

            if (dsExceptionReportUnitTest.Tables.Count > 0)
            {
                if (dsExceptionReportUnitTest.Tables[0].Rows.Count > 0)
                {
                    dsExceptionReportMain.Tables[0].Rows[0]["ExceptionUnitTestSuppress"] = false;
                }
                else
                {
                    dsExceptionReportMain.Tables[0].Rows[0]["ExceptionUnitTestSuppress"] = true;
                }
            }

            if (dsExceptionReportReturnIssuedUnit.Tables.Count > 0)
            {
                if (dsExceptionReportReturnIssuedUnit.Tables[0].Rows.Count > 0)
                {
                    dsExceptionReportMain.Tables[0].Rows[0][Common.VbecsTables.ExceptionReturnedUnit.TableName + "Suppress"] = false;
                }
                else
                {
                    dsExceptionReportMain.Tables[0].Rows[0][Common.VbecsTables.ExceptionReturnedUnit.TableName + "Suppress"] = true;
                }
            }

            if (dsExceptionReportTransfusion.Tables.Count > 0)
            {
                if (dsExceptionReportTransfusion.Tables[0].Rows.Count > 0)
                {
                    dsExceptionReportMain.Tables[0].Rows[0][Common.VbecsTables.ExceptionTransfusion.TableName + "Suppress"] = false;
                }
                else
                {
                    dsExceptionReportMain.Tables[0].Rows[0][Common.VbecsTables.ExceptionTransfusion.TableName + "Suppress"] = true;
                }
            }

            if (dsExceptionReportExpiredUnit.Tables.Count > 0)
            {
                if (dsExceptionReportExpiredUnit.Tables[0].Rows.Count > 0)
                {
                    dsExceptionReportMain.Tables[0].Rows[0][Common.VbecsTables.ExceptionExpiredUnitReceived.TableName + "Suppress"] = false;
                }
                else
                {
                    dsExceptionReportMain.Tables[0].Rows[0][Common.VbecsTables.ExceptionExpiredUnitReceived.TableName + "Suppress"] = true;
                }
            }

            if (dsExceptionReportUnitSupply.Tables.Count > 0)
            {
                if (dsExceptionReportUnitSupply.Tables[0].Rows.Count > 0)
                {
                    dsExceptionReportMain.Tables[0].Rows[0]["ExceptionReportUnitSupplySuppress"] = false;
                }
                else
                {
                    dsExceptionReportMain.Tables[0].Rows[0]["ExceptionReportUnitSupplySuppress"] = true;
                }
            }

            if (dsExceptionReportEmergencyIssueOrder.Tables.Count > 0)
            {
                if (dsExceptionReportEmergencyIssueOrder.Tables[0].Rows.Count > 0)
                {
                    dsExceptionReportMain.Tables[0].Rows[0]["ExceptionEmergencyIssueOrderSuppress"] = false;
                }
                else
                {
                    dsExceptionReportMain.Tables[0].Rows[0]["ExceptionEmergencyIssueOrderSuppress"] = true;
                }
            }

            if (dsExceptionReportUnitIssue.Tables.Count > 0)
            {
                if (dsExceptionReportUnitIssue.Tables[0].Rows.Count > 0)
                {
                    dsExceptionReportMain.Tables[0].Rows[0][Common.VbecsTables.ExceptionUnitIssue.TableName + "Suppress"] = false;
                }
                else
                {
                    dsExceptionReportMain.Tables[0].Rows[0][Common.VbecsTables.ExceptionUnitIssue.TableName + "Suppress"] = true;
                }
            }

            if (dsExceptionReportPatientRestriction.Tables.Count > 0)
            {
                if (dsExceptionReportPatientRestriction.Tables[0].Rows.Count > 0)
                {
                    dsExceptionReportMain.Tables[0].Rows[0][Common.VbecsTables.ExceptionPatientRestriction.TableName + "Suppress"] = false;
                }
                else
                {
                    dsExceptionReportMain.Tables[0].Rows[0][Common.VbecsTables.ExceptionPatientRestriction.TableName + "Suppress"] = true;
                }
            }

            if (dsExceptionReportEmergencyIssueUnitTest.Tables.Count > 0)
            {
                if (dsExceptionReportEmergencyIssueUnitTest.Tables[0].Rows.Count > 0)
                {
                    dsExceptionReportMain.Tables[0].Rows[0][Common.VbecsTables.ExceptionEmergencyIssueUnitTest.TableName + "Suppress"] = false;
                }
                else
                {
                    dsExceptionReportMain.Tables[0].Rows[0][Common.VbecsTables.ExceptionEmergencyIssueUnitTest.TableName + "Suppress"] = true;
                }
            }

            if (dsExceptionReportExpiredOrder.Tables.Count > 0)
            {
                if (dsExceptionReportExpiredOrder.Tables[0].Rows.Count > 0)
                {
                    dsExceptionReportMain.Tables[0].Rows[0][Common.VbecsTables.ExceptionExpiredTask.TableName + "Suppress"] = false;
                }
                else
                {
                    dsExceptionReportMain.Tables[0].Rows[0][Common.VbecsTables.ExceptionExpiredTask.TableName + "Suppress"] = true;
                }
            }

            if (dsExceptionReportRemoveFinalUnitStatus.Tables.Count > 0)
            {
                if (dsExceptionReportRemoveFinalUnitStatus.Tables[0].Rows.Count > 0)
                {
                    dsExceptionReportMain.Tables[0].Rows[0][Common.VbecsTables.ExceptionRemoveFinalUnitStatus.TableName + "Suppress"] = false;
                }
                else
                {
                    dsExceptionReportMain.Tables[0].Rows[0][Common.VbecsTables.ExceptionRemoveFinalUnitStatus.TableName + "Suppress"] = true;
                }
            }

            if (dsExceptionReportInconclusiveCrossmatch.Tables.Count > 0)
            {
                if (dsExceptionReportInconclusiveCrossmatch.Tables[0].Rows.Count > 0)
                {
                    dsExceptionReportMain.Tables[0].Rows[0][Common.VbecsTables.ExceptionInconclusiveCrossmatch.TableName + "Suppress"] = false;
                }
                else
                {
                    dsExceptionReportMain.Tables[0].Rows[0][Common.VbecsTables.ExceptionInconclusiveCrossmatch.TableName + "Suppress"] = true;
                }
            }

            if (dsExceptionReportElectronicTransfusion.Tables.Count > 0)
            {
                if (dsExceptionReportElectronicTransfusion.Tables[0].Rows.Count > 0)
                {
                    dsExceptionReportMain.Tables[0].Rows[0][Common.VbecsTables.ExceptionElectronicTransfusion.TableName + "Suppress"] = false;
                }
                else
                {
                    dsExceptionReportMain.Tables[0].Rows[0][Common.VbecsTables.ExceptionElectronicTransfusion.TableName + "Suppress"] = true;
                }
            }

            BOL.VbecsUser user = new BOL.VbecsUser(this.PrintedBy);
            DataSet dsHeader = BOL.VbecsReport.GetReportHeader(user.UserName);

            string reportCriteria = "Date Range: ";
            reportCriteria += Common.VBECSDateTime.FormatDateString(this.StartDate);
            reportCriteria += " - ";
            reportCriteria += Common.VBECSDateTime.FormatDateString(this.EndDate);
            DataSet dsFooter = BOL.VbecsReport.GetReportFooter(reportCriteria);

            ReportDataSource reportSourceMain = new ReportDataSource();
            if (_exceptionReports[Common.ExceptionReportType.ExceptionReportMain] != null &&
                ((DataSet)_exceptionReports[Common.ExceptionReportType.ExceptionReportMain]).Tables.Count > 0)
            {
                reportSourceMain.Name = "ExceptionMainDataSet";
                reportSourceMain.Value = ((DataSet)_exceptionReports[Common.ExceptionReportType.ExceptionReportMain]).Tables[0];
            }

            ReportDataSource reportSourceHeader = new ReportDataSource();
            if (dsHeader.Tables.Count > 0)
            {
                reportSourceHeader.Name = "ReportHeaderDataSet";
                reportSourceHeader.Value = dsHeader.Tables[0];
            }

            ReportDataSource reportSourceFooter = new ReportDataSource();
            if (dsFooter.Tables.Count > 0)
            {
                reportSourceFooter.Name = "ReportFooterDataSet";
                reportSourceFooter.Value = dsFooter.Tables[0];
            }

            Assembly assembly = Assembly.LoadFrom(Path.GetDirectoryName(Assembly.GetExecutingAssembly().Location) + "\\Reports.dll");
            Stream stream = assembly.GetManifestResourceStream("gov.va.med.vbecs.reports.ExceptionReport.ExceptionReport.rdlc");
            Stream streamExceptionQC = assembly.GetManifestResourceStream("gov.va.med.vbecs.reports.ExceptionReport.ExceptionQC.rdlc");
            Stream streamExceptionQCInactivated = assembly.GetManifestResourceStream("gov.va.med.vbecs.reports.ExceptionReport.ExceptionQCInactivated.rdlc");
            Stream streamExceptionQCNotPerformed = assembly.GetManifestResourceStream("gov.va.med.vbecs.reports.ExceptionReport.ExceptionQCNotPerformed.rdlc");
            Stream streamExceptionEmergencyIssueOrder = assembly.GetManifestResourceStream("gov.va.med.vbecs.reports.ExceptionReport.ExceptionEmergencyIssueOrder.rdlc");
            Stream streamExceptionQuarantine = assembly.GetManifestResourceStream("gov.va.med.vbecs.reports.ExceptionReport.ExceptionQuarantine.rdlc");
            Stream streamExceptionReflexTest = assembly.GetManifestResourceStream("gov.va.med.vbecs.reports.ExceptionReport.ExceptionReflexTest.rdlc");
            Stream streamExceptionSpecimen = assembly.GetManifestResourceStream("gov.va.med.vbecs.reports.ExceptionReport.ExceptionSpecimen.rdlc");
            Stream streamExceptionPatientAboRh = assembly.GetManifestResourceStream("gov.va.med.vbecs.reports.ExceptionReport.ExceptionPatientAboRh.rdlc");
            Stream streamExceptionPatientAboRhDiscrepancy = assembly.GetManifestResourceStream("gov.va.med.vbecs.reports.ExceptionReport.ExceptionPatientAboRhDiscrepancy.rdlc");
            Stream streamExceptionPatientAntigenDiscrepancy = assembly.GetManifestResourceStream("gov.va.med.vbecs.reports.ExceptionReport.ExceptionPatientAntigenDiscrepancy.rdlc");
            Stream streamExceptionPatientTest = assembly.GetManifestResourceStream("gov.va.med.vbecs.reports.ExceptionReport.ExceptionPatientTest.rdlc");
            Stream streamExceptionPatientTestInactivated = assembly.GetManifestResourceStream("gov.va.med.vbecs.reports.ExceptionReport.ExceptionPatientTestInactivated.rdlc");
            Stream streamExceptionUnitAboRh = assembly.GetManifestResourceStream("gov.va.med.vbecs.reports.ExceptionReport.ExceptionUnitAboRh.rdlc");
            Stream streamExceptionUnitModification = assembly.GetManifestResourceStream("gov.va.med.vbecs.reports.ExceptionReport.ExceptionUnitModification.rdlc");
            Stream streamExceptionUnitSelection = assembly.GetManifestResourceStream("gov.va.med.vbecs.reports.ExceptionReport.ExceptionUnitSelection.rdlc");
            Stream streamExceptionLabelMismatch = assembly.GetManifestResourceStream("gov.va.med.vbecs.reports.ExceptionReport.ExceptionLabelMismatch.rdlc");
            Stream streamExceptionCautionTagBTRFPrinting = assembly.GetManifestResourceStream("gov.va.med.vbecs.reports.ExceptionReport.ExceptionCautionTagBTRFPrinting.rdlc");
            Stream streamExceptionTrwSpecimen = assembly.GetManifestResourceStream("gov.va.med.vbecs.reports.ExceptionReport.ExceptionTrwSpecimen.rdlc");
            Stream streamExceptionTransfusionReactionWorkup = assembly.GetManifestResourceStream("gov.va.med.vbecs.reports.ExceptionReport.ExceptionTransfusionReactionWorkup.rdlc");
            Stream streamExceptionAntigenTest = assembly.GetManifestResourceStream("gov.va.med.vbecs.reports.ExceptionReport.ExceptionAntigenTest.rdlc");
            Stream streamExceptionUnitTest = assembly.GetManifestResourceStream("gov.va.med.vbecs.reports.ExceptionReport.ExceptionUnitTest.rdlc");
            Stream streamExceptionReturnIssuedUnit = assembly.GetManifestResourceStream("gov.va.med.vbecs.reports.ExceptionReport.ExceptionReturnIssuedUnit.rdlc");
            Stream streamExceptionTransfusion = assembly.GetManifestResourceStream("gov.va.med.vbecs.reports.ExceptionReport.ExceptionTransfusion.rdlc");
            Stream streamExceptionUnitSupply = assembly.GetManifestResourceStream("gov.va.med.vbecs.reports.ExceptionReport.ExceptionUnitSupply.rdlc");
            Stream streamExceptionExpiredUnit = assembly.GetManifestResourceStream("gov.va.med.vbecs.reports.ExceptionReport.ExceptionExpiredUnit.rdlc");
            Stream streamExceptionUnitIssue = assembly.GetManifestResourceStream("gov.va.med.vbecs.reports.ExceptionReport.ExceptionUnitIssue.rdlc");
            Stream streamExceptionPatientRestriction = assembly.GetManifestResourceStream("gov.va.med.vbecs.reports.ExceptionReport.ExceptionPatientRestriction.rdlc");
            Stream streamExceptionUnitERIssuedTestingProblem = assembly.GetManifestResourceStream("gov.va.med.vbecs.reports.ExceptionReport.ExceptionUnitERIssuedTestingProblem.rdlc");
            Stream streamExceptionExpiredOrder = assembly.GetManifestResourceStream("gov.va.med.vbecs.reports.ExceptionReport.ExceptionExpiredOrder.rdlc");
            Stream streamExceptionRemoveFinalUnitStatus = assembly.GetManifestResourceStream("gov.va.med.vbecs.reports.ExceptionReport.ExceptionRemoveFinalUnitStatus.rdlc");
            Stream streamExceptionInconclusiveCrossmatch = assembly.GetManifestResourceStream("gov.va.med.vbecs.reports.ExceptionReport.ExceptionInconclusiveCrossmatch.rdlc");
            Stream streamExceptionElectronicTransfusion = assembly.GetManifestResourceStream("gov.va.med.vbecs.reports.ExceptionReport.ExceptionElectronicTransfusion.rdlc");
            // LocalReport report = new LocalReport();

            LocalReport report = new LocalReport();
            report.SubreportProcessing += new SubreportProcessingEventHandler(LoadSubreportsEventHandler);

            report.DataSources.Add(reportSourceMain);
            report.DataSources.Add(reportSourceHeader);
            report.DataSources.Add(reportSourceFooter);
            report.LoadReportDefinition(stream);
            report.LoadSubreportDefinition("ExceptionQC", streamExceptionQC);
            report.LoadSubreportDefinition("ExceptionQCInactivated", streamExceptionQCInactivated);
            report.LoadSubreportDefinition("ExceptionQCNotPerformed", streamExceptionQCNotPerformed);
            report.LoadSubreportDefinition("ExceptionEmergencyIssueOrder", streamExceptionEmergencyIssueOrder);
            report.LoadSubreportDefinition("ExceptionQuarantine", streamExceptionQuarantine);
            report.LoadSubreportDefinition("ExceptionReflexTest", streamExceptionReflexTest);
            report.LoadSubreportDefinition("ExceptionSpecimen", streamExceptionSpecimen);
            report.LoadSubreportDefinition("ExceptionPatientAboRh", streamExceptionPatientAboRh);
            report.LoadSubreportDefinition("ExceptionPatientAboRhDiscrepancy", streamExceptionPatientAboRhDiscrepancy);
            report.LoadSubreportDefinition("ExceptionPatientAntigenDiscrepancy", streamExceptionPatientAntigenDiscrepancy);
            report.LoadSubreportDefinition("ExceptionPatientTest", streamExceptionPatientTest);
            report.LoadSubreportDefinition("ExceptionPatientTestInactivated", streamExceptionPatientTestInactivated);
            report.LoadSubreportDefinition("ExceptionUnitAboRh", streamExceptionUnitAboRh);
            report.LoadSubreportDefinition("ExceptionUnitModification", streamExceptionUnitModification);
            report.LoadSubreportDefinition("ExceptionUnitSelection", streamExceptionUnitSelection);
            report.LoadSubreportDefinition("ExceptionLabelMismatch", streamExceptionLabelMismatch);
            report.LoadSubreportDefinition("ExceptionCautionTagBTRFPrinting", streamExceptionCautionTagBTRFPrinting);
            report.LoadSubreportDefinition("ExceptionTrwSpecimen", streamExceptionTrwSpecimen);
            report.LoadSubreportDefinition("ExceptionTransfusionReactionWorkup", streamExceptionTransfusionReactionWorkup);
            report.LoadSubreportDefinition("ExceptionAntigenTest", streamExceptionAntigenTest);
            report.LoadSubreportDefinition("ExceptionUnitTest", streamExceptionUnitTest);
            report.LoadSubreportDefinition("ExceptionReturnIssuedUnit", streamExceptionReturnIssuedUnit);
            report.LoadSubreportDefinition("ExceptionTransfusion", streamExceptionTransfusion);
            report.LoadSubreportDefinition("ExceptionUnitSupply", streamExceptionUnitSupply);
            report.LoadSubreportDefinition("ExceptionExpiredUnit", streamExceptionExpiredUnit);
            report.LoadSubreportDefinition("ExceptionUnitIssue", streamExceptionUnitIssue);
            report.LoadSubreportDefinition("ExceptionPatientRestriction", streamExceptionPatientRestriction);
            report.LoadSubreportDefinition("ExceptionUnitERIssuedTestingProblem", streamExceptionUnitERIssuedTestingProblem);
            report.LoadSubreportDefinition("ExceptionExpiredOrder", streamExceptionExpiredOrder);
            report.LoadSubreportDefinition("ExceptionRemoveFinalUnitStatus", streamExceptionRemoveFinalUnitStatus);
            report.LoadSubreportDefinition("ExceptionInconclusiveCrossmatch", streamExceptionInconclusiveCrossmatch);
            report.LoadSubreportDefinition("ExceptionElectronicTransfusion", streamExceptionElectronicTransfusion);

            if (this.EndDate.Date.Equals(BOL.VBECSDateTime.GetDivisionCurrentDateTime().Date))
            {
                report.SetParameters(new ReportParameter("PreliminaryReport", "true"));
            }
            else
            {
                report.SetParameters(new ReportParameter("PreliminaryReport", "false"));
            }

            return new BOL.VbecsReportPrinter(this.PrinterName, true, report);
        }

        private void LoadSubreportsEventHandler(object sender, SubreportProcessingEventArgs e)
        {
            if (e.ReportPath == "ExceptionQC" || e.DataSourceNames[0].ToString() == "ExceptionQCDataSet")
            {
                ReportDataSource reportSourceExceptionQC = new ReportDataSource();
                if (_exceptionReports[Common.ExceptionReportType.ExceptionReportQC] != null &&
                   ((DataSet)_exceptionReports[Common.ExceptionReportType.ExceptionReportQC]).Tables.Count > 0)
                {
                    reportSourceExceptionQC.Name = "ExceptionQCDataSet";
                    reportSourceExceptionQC.Value = ((DataSet)_exceptionReports[Common.ExceptionReportType.ExceptionReportQC]).Tables[0];
                }
                e.DataSources.Add(reportSourceExceptionQC);
            }
            else if (e.ReportPath == "ExceptionQCInactivated" || e.DataSourceNames[0].ToString() == "ExceptionQCInactivatedDataSet")
            {
                ReportDataSource reportSourceExceptionQCInactivated = new ReportDataSource();
                if (_exceptionReports[Common.ExceptionReportType.ExceptionReportQCInactivated] != null &&
                   ((DataSet)_exceptionReports[Common.ExceptionReportType.ExceptionReportQCInactivated]).Tables.Count > 0)
                {
                    reportSourceExceptionQCInactivated.Name = "ExceptionQCInactivatedDataSet";
                    reportSourceExceptionQCInactivated.Value = ((DataSet)_exceptionReports[Common.ExceptionReportType.ExceptionReportQCInactivated]).Tables[0];
                }
                e.DataSources.Add(reportSourceExceptionQCInactivated);
            }
            else if (e.ReportPath == "ExceptionQCNotPerformed" || e.DataSourceNames[0].ToString() == "ExceptionQCNotPerformedDataSet")
            {
                ReportDataSource reportSourceExceptionQCNotPerformed = new ReportDataSource();
                if (_exceptionReports[Common.ExceptionReportType.ExceptionReportQCNotPerformed] != null &&
                   ((DataSet)_exceptionReports[Common.ExceptionReportType.ExceptionReportQCNotPerformed]).Tables.Count > 0)
                {
                    reportSourceExceptionQCNotPerformed.Name = "ExceptionQCNotPerformedDataSet";
                    reportSourceExceptionQCNotPerformed.Value = ((DataSet)_exceptionReports[Common.ExceptionReportType.ExceptionReportQCNotPerformed]).Tables[0];
                }
                e.DataSources.Add(reportSourceExceptionQCNotPerformed);
            }
            else if (e.ReportPath == "ExceptionEmergencyIssueOrder" || e.DataSourceNames[0].ToString() == "ExceptionEmergencyIssueOrderDataSet")
            {
                ReportDataSource reportSourceExceptionEmergencyIssueOrder = new ReportDataSource();
                if (_exceptionReports[Common.ExceptionReportType.ExceptionReportEmergencyIssueOrder] != null &&
                   ((DataSet)_exceptionReports[Common.ExceptionReportType.ExceptionReportEmergencyIssueOrder]).Tables.Count > 0)
                {
                    reportSourceExceptionEmergencyIssueOrder.Name = "ExceptionEmergencyIssueOrderDataSet";
                    reportSourceExceptionEmergencyIssueOrder.Value = ((DataSet)_exceptionReports[Common.ExceptionReportType.ExceptionReportEmergencyIssueOrder]).Tables[0];
                }
                e.DataSources.Add(reportSourceExceptionEmergencyIssueOrder);
            }
            else if (e.ReportPath == "ExceptionQuarantine" || e.DataSourceNames[0].ToString() == "ExceptionQuarantineDataSet")
            {
                ReportDataSource reportSourceExceptionQuarantine = new ReportDataSource();
                if (_exceptionReports[Common.ExceptionReportType.ExceptionReportQuarantine] != null &&
                   ((DataSet)_exceptionReports[Common.ExceptionReportType.ExceptionReportQuarantine]).Tables.Count > 0)
                {
                    reportSourceExceptionQuarantine.Name = "ExceptionQuarantineDataSet";
                    reportSourceExceptionQuarantine.Value = ((DataSet)_exceptionReports[Common.ExceptionReportType.ExceptionReportQuarantine]).Tables[0];
                }
                e.DataSources.Add(reportSourceExceptionQuarantine);
            }
            else if (e.ReportPath == "ExceptionReflexTest" || e.DataSourceNames[0].ToString() == "ExceptionReflexTestDataSet")
            {
                ReportDataSource reportSourceExceptionReflexTest = new ReportDataSource();
                if (_exceptionReports[Common.ExceptionReportType.ExceptionReportReflexTest] != null &&
                   ((DataSet)_exceptionReports[Common.ExceptionReportType.ExceptionReportReflexTest]).Tables.Count > 0)
                {
                    reportSourceExceptionReflexTest.Name = "ExceptionReflexTestDataSet";
                    reportSourceExceptionReflexTest.Value = ((DataSet)_exceptionReports[Common.ExceptionReportType.ExceptionReportReflexTest]).Tables[0];
                }
                e.DataSources.Add(reportSourceExceptionReflexTest);
            }
            else if (e.ReportPath == "ExceptionSpecimen" || e.DataSourceNames[0].ToString() == "ExceptionSpecimenDataSet")
            {
                ReportDataSource reportSourceExceptionSpecimen = new ReportDataSource();
                if (_exceptionReports[Common.ExceptionReportType.ExceptionReportSpecimen] != null &&
                   ((DataSet)_exceptionReports[Common.ExceptionReportType.ExceptionReportSpecimen]).Tables.Count > 0)
                {
                    reportSourceExceptionSpecimen.Name = "ExceptionSpecimenDataSet";
                    reportSourceExceptionSpecimen.Value = ((DataSet)_exceptionReports[Common.ExceptionReportType.ExceptionReportSpecimen]).Tables[0];
                }
                e.DataSources.Add(reportSourceExceptionSpecimen);
            }
            else if (e.ReportPath == "ExceptionPatientAboRh" || e.DataSourceNames[0].ToString() == "ExceptionPatientAboRhDataSet")
            {
                ReportDataSource reportSourceExceptionPatientAboRh = new ReportDataSource();
                if (_exceptionReports[Common.ExceptionReportType.ExceptionReportPatientAboRh] != null &&
                   ((DataSet)_exceptionReports[Common.ExceptionReportType.ExceptionReportPatientAboRh]).Tables.Count > 0)
                {
                    reportSourceExceptionPatientAboRh.Name = "ExceptionPatientAboRhDataSet";
                    reportSourceExceptionPatientAboRh.Value = ((DataSet)_exceptionReports[Common.ExceptionReportType.ExceptionReportPatientAboRh]).Tables[0];
                }
                e.DataSources.Add(reportSourceExceptionPatientAboRh);
            }
            else if (e.ReportPath == "ExceptionPatientAboRhDiscrepancy" || e.DataSourceNames[0].ToString() == "ExceptionPatientAboRhDiscrepancyDataSet")
            {
                ReportDataSource reportSourceExceptionPatientAboRhDiscrepancy = new ReportDataSource();
                if (_exceptionReports[Common.ExceptionReportType.ExceptionReportPatientAboRhDiscrepancy] != null &&
                   ((DataSet)_exceptionReports[Common.ExceptionReportType.ExceptionReportPatientAboRhDiscrepancy]).Tables.Count > 0)
                {
                    reportSourceExceptionPatientAboRhDiscrepancy.Name = "ExceptionPatientAboRhDiscrepancyDataSet";
                    reportSourceExceptionPatientAboRhDiscrepancy.Value = ((DataSet)_exceptionReports[Common.ExceptionReportType.ExceptionReportPatientAboRhDiscrepancy]).Tables[0];
                }
                e.DataSources.Add(reportSourceExceptionPatientAboRhDiscrepancy);
            }
            else if (e.ReportPath == "ExceptionPatientAntigenDiscrepancy" || e.DataSourceNames[0].ToString() == "ExceptionPatientAntigenDiscrepancyDataSet")
            {
                ReportDataSource reportSourceExceptionPatientAntigenDiscrepancy = new ReportDataSource();
                if (_exceptionReports[Common.ExceptionReportType.ExceptionReportPatientAntigenDiscrepancy] != null &&
                   ((DataSet)_exceptionReports[Common.ExceptionReportType.ExceptionReportPatientAntigenDiscrepancy]).Tables.Count > 0)
                {
                    reportSourceExceptionPatientAntigenDiscrepancy.Name = "ExceptionPatientAntigenDiscrepancyDataSet";
                    reportSourceExceptionPatientAntigenDiscrepancy.Value = ((DataSet)_exceptionReports[Common.ExceptionReportType.ExceptionReportPatientAntigenDiscrepancy]).Tables[0];
                }
                e.DataSources.Add(reportSourceExceptionPatientAntigenDiscrepancy);
            }
            else if (e.ReportPath == "ExceptionPatientTest" || e.DataSourceNames[0].ToString() == "ExceptionPatientTestDataSet")
            {
                ReportDataSource reportSourceExceptionPatientTest = new ReportDataSource();
                if (_exceptionReports[Common.ExceptionReportType.ExceptionReportPatientTest] != null &&
                   ((DataSet)_exceptionReports[Common.ExceptionReportType.ExceptionReportPatientTest]).Tables.Count > 0)
                {
                    reportSourceExceptionPatientTest.Name = "ExceptionPatientTestDataSet";
                    reportSourceExceptionPatientTest.Value = ((DataSet)_exceptionReports[Common.ExceptionReportType.ExceptionReportPatientTest]).Tables[0];
                }
                e.DataSources.Add(reportSourceExceptionPatientTest);
            }
            else if (e.ReportPath == "ExceptionPatientTestInactivated" || e.DataSourceNames[0].ToString() == "ExceptionPatientTestInactivatedDataSet")
            {
                ReportDataSource reportSourceExceptionPatientTestInactivated = new ReportDataSource();
                if (_exceptionReports[Common.ExceptionReportType.ExceptionReportPatientTestInactivated] != null &&
                   ((DataSet)_exceptionReports[Common.ExceptionReportType.ExceptionReportPatientTestInactivated]).Tables.Count > 0)
                {
                    reportSourceExceptionPatientTestInactivated.Name = "ExceptionPatientTestInactivatedDataSet";
                    reportSourceExceptionPatientTestInactivated.Value = ((DataSet)_exceptionReports[Common.ExceptionReportType.ExceptionReportPatientTestInactivated]).Tables[0];
                }
                e.DataSources.Add(reportSourceExceptionPatientTestInactivated);
            }
            else if (e.ReportPath == "ExceptionUnitAboRh" || e.DataSourceNames[0].ToString() == "ExceptionUnitAboRhDataSet")
            {
                ReportDataSource reportSourceExceptionUnitAboRh = new ReportDataSource();
                if (_exceptionReports[Common.ExceptionReportType.ExceptionReportUnitAboRh] != null &&
                   ((DataSet)_exceptionReports[Common.ExceptionReportType.ExceptionReportUnitAboRh]).Tables.Count > 0)
                {
                    reportSourceExceptionUnitAboRh.Name = "ExceptionUnitAboRhDataSet";
                    reportSourceExceptionUnitAboRh.Value = ((DataSet)_exceptionReports[Common.ExceptionReportType.ExceptionReportUnitAboRh]).Tables[0];
                }
                e.DataSources.Add(reportSourceExceptionUnitAboRh);
            }
            else if (e.ReportPath == "ExceptionUnitModification" || e.DataSourceNames[0].ToString() == "ExceptionUnitModificationDataSet")
            {
                ReportDataSource reportSourceExceptionUnitModification = new ReportDataSource();
                if (_exceptionReports[Common.ExceptionReportType.ExceptionReportUnitModification] != null &&
                   ((DataSet)_exceptionReports[Common.ExceptionReportType.ExceptionReportUnitModification]).Tables.Count > 0)
                {
                    reportSourceExceptionUnitModification.Name = "ExceptionUnitModificationDataSet";
                    reportSourceExceptionUnitModification.Value = ((DataSet)_exceptionReports[Common.ExceptionReportType.ExceptionReportUnitModification]).Tables[0];
                }
                e.DataSources.Add(reportSourceExceptionUnitModification);
            }
            else if (e.ReportPath == "ExceptionUnitSelection" || e.DataSourceNames[0].ToString() == "ExceptionUnitSelectionDataSet")
            {
                ReportDataSource reportSourceExceptionUnitSelection = new ReportDataSource();
                if (_exceptionReports[Common.ExceptionReportType.ExceptionReportUnitSelection] != null &&
                   ((DataSet)_exceptionReports[Common.ExceptionReportType.ExceptionReportUnitSelection]).Tables.Count > 0)
                {
                    reportSourceExceptionUnitSelection.Name = "ExceptionUnitSelectionDataSet";
                    reportSourceExceptionUnitSelection.Value = ((DataSet)_exceptionReports[Common.ExceptionReportType.ExceptionReportUnitSelection]).Tables[0];
                }
                e.DataSources.Add(reportSourceExceptionUnitSelection);
            }
            else if (e.ReportPath == "ExceptionLabelMismatch" || e.DataSourceNames[0].ToString() == "ExceptionLabelMismatchDataSet")
            {
                ReportDataSource reportSourceExceptionLabelMismatch = new ReportDataSource();
                if (_exceptionReports[Common.ExceptionReportType.ExceptionReportLabelMismatch] != null &&
                   ((DataSet)_exceptionReports[Common.ExceptionReportType.ExceptionReportLabelMismatch]).Tables.Count > 0)
                {
                    reportSourceExceptionLabelMismatch.Name = "ExceptionLabelMismatchDataSet";
                    reportSourceExceptionLabelMismatch.Value = ((DataSet)_exceptionReports[Common.ExceptionReportType.ExceptionReportLabelMismatch]).Tables[0];
                }
                e.DataSources.Add(reportSourceExceptionLabelMismatch);
            }
            else if (e.ReportPath == "ExceptionCautionTagBTRFPrinting" || e.DataSourceNames[0].ToString() == "ExceptionCautionTagBTRFPrintingDataSet")
            {
                ReportDataSource reportSourceExceptionCautionTagBTRFPrinting = new ReportDataSource();
                if (_exceptionReports[Common.ExceptionReportType.ExceptionReportCautionTagBTRFPrinting] != null &&
                   ((DataSet)_exceptionReports[Common.ExceptionReportType.ExceptionReportCautionTagBTRFPrinting]).Tables.Count > 0)
                {
                    reportSourceExceptionCautionTagBTRFPrinting.Name = "ExceptionCautionTagBTRFPrintingDataSet";
                    reportSourceExceptionCautionTagBTRFPrinting.Value = ((DataSet)_exceptionReports[Common.ExceptionReportType.ExceptionReportCautionTagBTRFPrinting]).Tables[0];
                }
                e.DataSources.Add(reportSourceExceptionCautionTagBTRFPrinting);
            }
            else if (e.ReportPath == "ExceptionTrwSpecimen" || e.DataSourceNames[0].ToString() == "ExceptionTrwSpecimenDataSet")
            {
                ReportDataSource reportSourceExceptionTrwSpecimen = new ReportDataSource();
                if (_exceptionReports[Common.ExceptionReportType.ExceptionReportTrwSpecimen] != null &&
                   ((DataSet)_exceptionReports[Common.ExceptionReportType.ExceptionReportTrwSpecimen]).Tables.Count > 0)
                {
                    reportSourceExceptionTrwSpecimen.Name = "ExceptionTrwSpecimenDataSet";
                    reportSourceExceptionTrwSpecimen.Value = ((DataSet)_exceptionReports[Common.ExceptionReportType.ExceptionReportTrwSpecimen]).Tables[0];
                }
                e.DataSources.Add(reportSourceExceptionTrwSpecimen);
            }
            else if (e.ReportPath == "ExceptionTransfusionReactionWorkup" || e.DataSourceNames[0].ToString() == "ExceptionTransfusionReactionWorkupDataSet")
            {
                ReportDataSource reportSourceExceptionTransfusionReactionWorkup = new ReportDataSource();
                if (_exceptionReports[Common.ExceptionReportType.ExceptionReportTransfusionReactionWorkup] != null &&
                   ((DataSet)_exceptionReports[Common.ExceptionReportType.ExceptionReportTransfusionReactionWorkup]).Tables.Count > 0)
                {
                    reportSourceExceptionTransfusionReactionWorkup.Name = "ExceptionTransfusionReactionWorkupDataSet";
                    reportSourceExceptionTransfusionReactionWorkup.Value = ((DataSet)_exceptionReports[Common.ExceptionReportType.ExceptionReportTransfusionReactionWorkup]).Tables[0];
                }
                e.DataSources.Add(reportSourceExceptionTransfusionReactionWorkup);
            }
            else if (e.ReportPath == "ExceptionAntigenTest" || e.DataSourceNames[0].ToString() == "ExceptionAntigenTestDataSet")
            {
                ReportDataSource reportSourceExceptionAntigenTest = new ReportDataSource();
                if (_exceptionReports[Common.ExceptionReportType.ExceptionReportAntigenTest] != null &&
                   ((DataSet)_exceptionReports[Common.ExceptionReportType.ExceptionReportAntigenTest]).Tables.Count > 0)
                {
                    reportSourceExceptionAntigenTest.Name = "ExceptionAntigenTestDataSet";
                    reportSourceExceptionAntigenTest.Value = ((DataSet)_exceptionReports[Common.ExceptionReportType.ExceptionReportAntigenTest]).Tables[0];
                }
                e.DataSources.Add(reportSourceExceptionAntigenTest);
            }
            else if (e.ReportPath == "ExceptionUnitTest" || e.DataSourceNames[0].ToString() == "ExceptionUnitTestDataSet")
            {
                ReportDataSource reportSourceExceptionUnitTest = new ReportDataSource();
                if (_exceptionReports[Common.ExceptionReportType.ExceptionReportUnitTest] != null &&
                   ((DataSet)_exceptionReports[Common.ExceptionReportType.ExceptionReportUnitTest]).Tables.Count > 0)
                {
                    reportSourceExceptionUnitTest.Name = "ExceptionUnitTestDataSet";
                    reportSourceExceptionUnitTest.Value = ((DataSet)_exceptionReports[Common.ExceptionReportType.ExceptionReportUnitTest]).Tables[0];
                }
                e.DataSources.Add(reportSourceExceptionUnitTest);
            }
            else if (e.ReportPath == "ExceptionReturnIssuedUnit" || e.DataSourceNames[0].ToString() == "ExceptionReturnIssuedUnitDataSet")
            {
                ReportDataSource reportSourceExceptionReturnIssuedUnit = new ReportDataSource();
                if (_exceptionReports[Common.ExceptionReportType.ExceptionReportReturnIssuedUnit] != null &&
                   ((DataSet)_exceptionReports[Common.ExceptionReportType.ExceptionReportReturnIssuedUnit]).Tables.Count > 0)
                {
                    reportSourceExceptionReturnIssuedUnit.Name = "ExceptionReturnIssuedUnitDataSet";
                    reportSourceExceptionReturnIssuedUnit.Value = ((DataSet)_exceptionReports[Common.ExceptionReportType.ExceptionReportReturnIssuedUnit]).Tables[0];
                }
                e.DataSources.Add(reportSourceExceptionReturnIssuedUnit);
            }
            else if (e.ReportPath == "ExceptionTransfusion" || e.DataSourceNames[0].ToString() == "ExceptionTransfusionDataSet")
            {
                ReportDataSource reportSourceExceptionTransfusion = new ReportDataSource();
                if (_exceptionReports[Common.ExceptionReportType.ExceptionReportTransfusion] != null &&
                   ((DataSet)_exceptionReports[Common.ExceptionReportType.ExceptionReportTransfusion]).Tables.Count > 0)
                {
                    reportSourceExceptionTransfusion.Name = "ExceptionTransfusionDataSet";
                    reportSourceExceptionTransfusion.Value = ((DataSet)_exceptionReports[Common.ExceptionReportType.ExceptionReportTransfusion]).Tables[0];
                }
                e.DataSources.Add(reportSourceExceptionTransfusion);
            }
            else if (e.ReportPath == "ExceptionUnitSupply" || e.DataSourceNames[0].ToString() == "ExceptionUnitSupplyDataSet")
            {
                ReportDataSource reportSourceExceptionUnitSupply = new ReportDataSource();
                if (_exceptionReports[Common.ExceptionReportType.ExceptionReportUnitSupply] != null &&
                   ((DataSet)_exceptionReports[Common.ExceptionReportType.ExceptionReportUnitSupply]).Tables.Count > 0)
                {
                    reportSourceExceptionUnitSupply.Name = "ExceptionUnitSupplyDataSet";
                    reportSourceExceptionUnitSupply.Value = ((DataSet)_exceptionReports[Common.ExceptionReportType.ExceptionReportUnitSupply]).Tables[0];
                }
                e.DataSources.Add(reportSourceExceptionUnitSupply);
            }
            else if (e.ReportPath == "ExceptionExpiredUnit" || e.DataSourceNames[0].ToString() == "ExceptionExpiredUnitDataSet")
            {
                ReportDataSource reportSourceExceptionExpiredUnit = new ReportDataSource();
                if (_exceptionReports[Common.ExceptionReportType.ExceptionReportExpiredUnitReceived] != null &&
                   ((DataSet)_exceptionReports[Common.ExceptionReportType.ExceptionReportExpiredUnitReceived]).Tables.Count > 0)
                {
                    reportSourceExceptionExpiredUnit.Name = "ExceptionExpiredUnitDataSet";
                    reportSourceExceptionExpiredUnit.Value = ((DataSet)_exceptionReports[Common.ExceptionReportType.ExceptionReportExpiredUnitReceived]).Tables[0];
                }
                e.DataSources.Add(reportSourceExceptionExpiredUnit);
            }
            else if (e.ReportPath == "ExceptionUnitIssue" || e.DataSourceNames[0].ToString() == "ExceptionUnitIssueDataSet")
            {
                ReportDataSource reportSourceExceptionUnitIssue = new ReportDataSource();
                if (_exceptionReports[Common.ExceptionReportType.ExceptionReportUnitIssue] != null &&
                   ((DataSet)_exceptionReports[Common.ExceptionReportType.ExceptionReportUnitIssue]).Tables.Count > 0)
                {
                    reportSourceExceptionUnitIssue.Name = "ExceptionUnitIssueDataSet";
                    reportSourceExceptionUnitIssue.Value = ((DataSet)_exceptionReports[Common.ExceptionReportType.ExceptionReportUnitIssue]).Tables[0];
                }
                e.DataSources.Add(reportSourceExceptionUnitIssue);
            }
            else if (e.ReportPath == "ExceptionPatientRestriction" || e.DataSourceNames[0].ToString() == "ExceptionPatientRestrictionDataSet")
            {
                ReportDataSource reportSourceExceptionPatientRestriction = new ReportDataSource();
                if (_exceptionReports[Common.ExceptionReportType.ExceptionReportPatientRestriction] != null &&
                   ((DataSet)_exceptionReports[Common.ExceptionReportType.ExceptionReportPatientRestriction]).Tables.Count > 0)
                {
                    reportSourceExceptionPatientRestriction.Name = "ExceptionPatientRestrictionDataSet";
                    reportSourceExceptionPatientRestriction.Value = ((DataSet)_exceptionReports[Common.ExceptionReportType.ExceptionReportPatientRestriction]).Tables[0];
                }
                e.DataSources.Add(reportSourceExceptionPatientRestriction);
            }
            else if (e.ReportPath == "ExceptionUnitERIssuedTestingProblem" || e.DataSourceNames[0].ToString() == "ExceptionUnitERIssuedTestingProblemDataSet")
            {
                ReportDataSource reportSourceExceptionUnitERIssuedTestingProblem = new ReportDataSource();
                if (_exceptionReports[Common.ExceptionReportType.ExceptionReportUnitERIssuedTestingProblem] != null &&
                   ((DataSet)_exceptionReports[Common.ExceptionReportType.ExceptionReportUnitERIssuedTestingProblem]).Tables.Count > 0)
                {
                    reportSourceExceptionUnitERIssuedTestingProblem.Name = "ExceptionUnitERIssuedTestingProblemDataSet";
                    reportSourceExceptionUnitERIssuedTestingProblem.Value = ((DataSet)_exceptionReports[Common.ExceptionReportType.ExceptionReportUnitERIssuedTestingProblem]).Tables[0];
                }
                e.DataSources.Add(reportSourceExceptionUnitERIssuedTestingProblem);
            }
            else if (e.ReportPath == "ExceptionExpiredOrder" || e.DataSourceNames[0].ToString() == "ExceptionExpiredOrderDataSet")
            {
                ReportDataSource reportSourceExceptionExpiredOrder = new ReportDataSource();
                if (_exceptionReports[Common.ExceptionReportType.ExceptionReportExpiredTaskProcessed] != null &&
                   ((DataSet)_exceptionReports[Common.ExceptionReportType.ExceptionReportExpiredTaskProcessed]).Tables.Count > 0)
                {
                    reportSourceExceptionExpiredOrder.Name = "ExceptionExpiredOrderDataSet";
                    reportSourceExceptionExpiredOrder.Value = ((DataSet)_exceptionReports[Common.ExceptionReportType.ExceptionReportExpiredTaskProcessed]).Tables[0];
                }
                e.DataSources.Add(reportSourceExceptionExpiredOrder);
            }
            else if (e.ReportPath == "ExceptionRemoveFinalUnitStatus" || e.DataSourceNames[0].ToString() == "ExceptionRemoveFinalUnitStatusDataSet")
            {
                ReportDataSource reportSourceExceptionRemoveFinalUnitStatus = new ReportDataSource();
                if (_exceptionReports[Common.ExceptionReportType.ExceptionReportRemoveFinalUnitStatus] != null &&
                   ((DataSet)_exceptionReports[Common.ExceptionReportType.ExceptionReportRemoveFinalUnitStatus]).Tables.Count > 0)
                {
                    reportSourceExceptionRemoveFinalUnitStatus.Name = "ExceptionRemoveFinalUnitStatusDataSet";
                    reportSourceExceptionRemoveFinalUnitStatus.Value = ((DataSet)_exceptionReports[Common.ExceptionReportType.ExceptionReportRemoveFinalUnitStatus]).Tables[0];
                }
                e.DataSources.Add(reportSourceExceptionRemoveFinalUnitStatus);
            }
            else if (e.ReportPath == "ExceptionInconclusiveCrossmatch" || e.DataSourceNames[0].ToString() == "ExceptionInconclusiveCrossmatchDataSet")
            {
                ReportDataSource reportSourceExceptionInconclusiveCrossmatch = new ReportDataSource();
                if (_exceptionReports[Common.ExceptionReportType.ExceptionReportCrossmatch] != null &&
                   ((DataSet)_exceptionReports[Common.ExceptionReportType.ExceptionReportCrossmatch]).Tables.Count > 0)
                {
                    reportSourceExceptionInconclusiveCrossmatch.Name = "ExceptionInconclusiveCrossmatchDataSet";
                    reportSourceExceptionInconclusiveCrossmatch.Value = ((DataSet)_exceptionReports[Common.ExceptionReportType.ExceptionReportCrossmatch]).Tables[0];
                }
                e.DataSources.Add(reportSourceExceptionInconclusiveCrossmatch);
            }
            else if (e.ReportPath == "ExceptionElectronicTransfusion" || e.DataSourceNames[0].ToString() == "ExceptionElectronicTransfusionDataSet")
            {
                ReportDataSource reportSourceExceptionElectronicTransfusion = new ReportDataSource();
                if (_exceptionReports[Common.ExceptionReportType.ExceptionReportElectronicTransfusion] != null &&
                   ((DataSet)_exceptionReports[Common.ExceptionReportType.ExceptionReportElectronicTransfusion]).Tables.Count > 0)
                {
                    reportSourceExceptionElectronicTransfusion.Name = "ExceptionElectronicTransfusionDataSet";
                    reportSourceExceptionElectronicTransfusion.Value = ((DataSet)_exceptionReports[Common.ExceptionReportType.ExceptionReportElectronicTransfusion]).Tables[0];
                }
                e.DataSources.Add(reportSourceExceptionElectronicTransfusion);
            }
        }


		#endregion
	}
}
